/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.jdbc;

import java.lang.ref.Reference;
import java.lang.ref.ReferenceQueue;
import java.lang.ref.WeakReference;
import java.net.URI;
import java.sql.Array;
import java.sql.Blob;
import java.sql.CallableStatement;
import java.sql.ClientInfoStatus;
import java.sql.Clob;
import java.sql.DatabaseMetaData;
import java.sql.NClob;
import java.sql.PreparedStatement;
import java.sql.SQLClientInfoException;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.sql.SQLWarning;
import java.sql.SQLXML;
import java.sql.Savepoint;
import java.sql.Statement;
import java.sql.Struct;
import java.time.Clock;
import java.time.Duration;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.function.UnaryOperator;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.neo4j.jdbc.ArrayImpl;
import org.neo4j.jdbc.AuthenticationManager;
import org.neo4j.jdbc.BoltConnectionObservations;
import org.neo4j.jdbc.BookmarkManager;
import org.neo4j.jdbc.CallableStatementImpl;
import org.neo4j.jdbc.DatabaseMetadataImpl;
import org.neo4j.jdbc.DefaultAuthenticationManagerImpl;
import org.neo4j.jdbc.DefaultTransactionImpl;
import org.neo4j.jdbc.Events;
import org.neo4j.jdbc.Lazy;
import org.neo4j.jdbc.Neo4jConnection;
import org.neo4j.jdbc.Neo4jConversions;
import org.neo4j.jdbc.Neo4jDatabaseMetaData;
import org.neo4j.jdbc.Neo4jDriver;
import org.neo4j.jdbc.Neo4jException;
import org.neo4j.jdbc.Neo4jTransaction;
import org.neo4j.jdbc.PreparedStatementImpl;
import org.neo4j.jdbc.ProductVersion;
import org.neo4j.jdbc.StatementImpl;
import org.neo4j.jdbc.Tracing;
import org.neo4j.jdbc.Warnings;
import org.neo4j.jdbc.authn.spi.Authentication;
import org.neo4j.jdbc.events.ConnectionListener;
import org.neo4j.jdbc.events.StatementListener;
import org.neo4j.jdbc.internal.shaded.bolt.AccessMode;
import org.neo4j.jdbc.internal.shaded.bolt.BasicResponseHandler;
import org.neo4j.jdbc.internal.shaded.bolt.BoltConnection;
import org.neo4j.jdbc.internal.shaded.bolt.ResponseHandler;
import org.neo4j.jdbc.internal.shaded.bolt.exception.BoltConnectionReadTimeoutException;
import org.neo4j.jdbc.internal.shaded.bolt.exception.BoltFailureException;
import org.neo4j.jdbc.internal.shaded.bolt.message.Messages;
import org.neo4j.jdbc.internal.shaded.bolt.observation.ImmutableObservation;
import org.neo4j.jdbc.tracing.Neo4jTracer;
import org.neo4j.jdbc.translator.spi.Cache;
import org.neo4j.jdbc.translator.spi.Translator;
import org.neo4j.jdbc.translator.spi.View;

final class ConnectionImpl
implements Neo4jConnection {
    private static final Pattern PATTERN_ENFORCE_CYPHER = Pattern.compile("(['`\"])?[^'`\"]*/\\*\\+ NEO4J FORCE_CYPHER \\*/[^'`\"]*(['`\"])?");
    static final Logger LOGGER = Logger.getLogger("org.neo4j.jdbc.connection");
    private static final int TRANSLATION_CACHE_SIZE = 128;
    private final URI databaseUrl;
    private final BoltConnection boltConnection;
    private final Lazy<BoltConnection> boltConnectionForMetaData;
    private final Lazy<DatabaseMetaData> databaseMetadData;
    private final Set<Reference<Statement>> trackedStatementReferences = new HashSet<Reference<Statement>>();
    private final ReferenceQueue<Statement> trackedStatementReferenceQueue = new ReferenceQueue();
    private final Lazy<List<Translator>> translators;
    private final boolean enableSqlTranslation;
    private final boolean enableTranslationCaching;
    private final boolean rewriteBatchedStatements;
    private final boolean rewritePlaceholders;
    private Neo4jTransaction transaction;
    private boolean autoCommit = true;
    private boolean readOnly;
    private int networkTimeout;
    private final Warnings warnings = new Warnings();
    private SQLException fatalException;
    private boolean closed;
    private final Cache<String, String> l2cache = Cache.getInstance(128);
    private final BookmarkManager bookmarkManager;
    private final AuthenticationManager authenticationManager;
    private final Map<String, Object> transactionMetadata = new ConcurrentHashMap<String, Object>();
    private final int relationshipSampleSize;
    private final String databaseName;
    private final AtomicBoolean resetNeeded = new AtomicBoolean(false);
    private final Map<String, String> clientInfo = new ConcurrentHashMap<String, String>();
    private final Consumer<Boolean> onClose;
    private final Set<ConnectionListener> listeners = new HashSet<ConnectionListener>();

    ConnectionImpl(URI databaseUrl, Supplier<Authentication> authenticationSupplier, Function<Authentication, BoltConnection> boltConnectionSupplier, Supplier<List<Translator>> translators, boolean enableSQLTranslation, boolean enableTranslationCaching, boolean rewriteBatchedStatements, boolean rewritePlaceholders, BookmarkManager bookmarkManager, Map<String, Object> transactionMetadata, int relationshipSampleSize, String databaseName, Consumer<Boolean> onClose, List<ConnectionListener> initalListeners) {
        Objects.requireNonNull(boltConnectionSupplier);
        this.databaseUrl = Objects.requireNonNull(databaseUrl);
        this.authenticationManager = new DefaultAuthenticationManagerImpl(this.databaseUrl, authenticationSupplier, Clock.systemUTC(), Duration.ofSeconds(0L));
        initalListeners.forEach(this::addListener);
        this.boltConnection = boltConnectionSupplier.apply(this.authenticationManager.getOrRefresh());
        this.boltConnectionForMetaData = Lazy.of(() -> (BoltConnection)boltConnectionSupplier.apply(this.authenticationManager.getOrRefresh()));
        this.translators = Lazy.of(translators::get);
        this.enableSqlTranslation = enableSQLTranslation;
        this.enableTranslationCaching = enableTranslationCaching;
        this.rewriteBatchedStatements = rewriteBatchedStatements;
        this.rewritePlaceholders = rewritePlaceholders;
        this.bookmarkManager = Objects.requireNonNull(bookmarkManager);
        this.transactionMetadata.putAll(Objects.requireNonNullElseGet(transactionMetadata, Map::of));
        this.relationshipSampleSize = relationshipSampleSize;
        this.databaseName = Objects.requireNonNull(databaseName);
        this.databaseMetadData = Lazy.of(() -> {
            List<View> views = this.translators.resolve().stream().flatMap(t -> t.getViews().stream()).toList();
            return new DatabaseMetadataImpl(this, this.enableSqlTranslation, this.relationshipSampleSize, views);
        });
        this.onClose = Objects.requireNonNullElse(onClose, aborted -> {});
    }

    void notifyStatementListeners(Class<? extends Statement> type) {
        ConnectionListener.StatementClosedEvent event = new ConnectionListener.StatementClosedEvent(this.databaseUrl, type);
        Events.notify(this.listeners, listener -> listener.onStatementClosed(event));
    }

    UnaryOperator<String> getTranslator(Consumer<SQLWarning> warningConsumer) throws SQLException {
        return this.getTranslator(false, warningConsumer);
    }

    UnaryOperator<String> getTranslator(boolean force, Consumer<SQLWarning> warningConsumer) throws SQLException {
        List<Translator> resolvedTranslators = !this.enableSqlTranslation && !force ? List.of((statement, optionalDatabaseMetaData) -> statement) : this.translators.resolve();
        if (resolvedTranslators.isEmpty()) {
            throw Neo4jDriver.noTranslatorsAvailableException();
        }
        DatabaseMetaData metaData = this.getMetaData();
        TranslatorChain sqlTranslator = new TranslatorChain(resolvedTranslators, metaData, warningConsumer);
        if (this.enableTranslationCaching) {
            return sql -> {
                ConnectionImpl connectionImpl = this;
                synchronized (connectionImpl) {
                    if (this.l2cache.containsKey(sql)) {
                        return this.l2cache.get(sql);
                    }
                    String translation = sqlTranslator.apply((String)sql);
                    this.l2cache.put((String)sql, translation);
                    ConnectionListener.TranslationCachedEvent event = new ConnectionListener.TranslationCachedEvent(this.l2cache.size());
                    Events.notify(this.listeners, listener -> listener.onTranslationCached(event));
                    return translation;
                }
            };
        }
        return sqlTranslator;
    }

    @Override
    public Statement createStatement() throws SQLException {
        return this.createStatement(1003, 1007, 2);
    }

    @Override
    public PreparedStatement prepareStatement(String sql) throws SQLException {
        return this.prepareStatement(sql, 2);
    }

    @Override
    public CallableStatement prepareCall(String sql) throws SQLException {
        return this.prepareCall(sql, 1003, 1007, 2);
    }

    @Override
    public String nativeSQL(String sql) throws SQLException {
        LOGGER.log(Level.FINER, () -> "Translating `%s` into native SQL".formatted(sql));
        this.assertIsOpen();
        return (String)this.getTranslator(true, this.warnings).apply(sql);
    }

    @Override
    public void setAutoCommit(boolean autoCommit) throws SQLException {
        LOGGER.log(Level.FINER, () -> "Setting auto commit to %s".formatted(autoCommit));
        this.assertIsOpen();
        if (this.autoCommit == autoCommit) {
            return;
        }
        if (this.transaction != null) {
            if (Neo4jTransaction.State.OPEN_FAILED == this.transaction.getState()) {
                throw new Neo4jException(Neo4jException.withReason("The existing transaction must be rolled back explicitly"));
            }
            if (this.transaction.isRunnable()) {
                this.transaction.commit();
            }
        }
        this.autoCommit = autoCommit;
    }

    @Override
    public boolean getAutoCommit() throws SQLException {
        LOGGER.log(Level.FINER, () -> "Getting auto commit");
        this.assertIsOpen();
        return this.autoCommit;
    }

    @Override
    public void commit() throws SQLException {
        LOGGER.log(Level.FINER, () -> "Commiting");
        this.assertIsOpen();
        if (this.transaction == null || Neo4jTransaction.State.COMMITTED.equals((Object)this.transaction.getState())) {
            LOGGER.log(Level.INFO, "There is no active transaction that can be committed, ignoring");
            return;
        }
        if (this.transaction.isAutoCommit()) {
            throw new Neo4jException(Neo4jException.GQLError.$2DN01.withTemplatedMessage("Auto commit transaction may not be managed explicitly"));
        }
        this.transaction.commit();
        this.transaction = null;
    }

    @Override
    public void rollback() throws SQLException {
        LOGGER.log(Level.FINER, () -> "Rolling back");
        this.assertIsOpen();
        if (this.transaction == null || !this.transaction.isRunnable()) {
            LOGGER.log(Level.INFO, "There is no active transaction that can be rolled back, ignoring");
            this.transaction = null;
            return;
        }
        if (this.transaction.isAutoCommit()) {
            throw new Neo4jException(Neo4jException.GQLError.$40N01.withTemplatedMessage("Auto commit transaction may not be managed explicitly"));
        }
        this.transaction.rollback();
        this.transaction = null;
    }

    @Override
    public void close() throws SQLException {
        LOGGER.log(Level.FINER, () -> "Closing");
        if (this.isClosed()) {
            return;
        }
        Exception exceptionDuringRollback = null;
        if (this.transaction != null && this.transaction.isRunnable()) {
            try {
                this.transaction.rollback();
            }
            catch (Exception ex) {
                exceptionDuringRollback = ex;
            }
        }
        try {
            this.closeBoltConnections();
        }
        catch (Exception ex) {
            if (ex instanceof InterruptedException) {
                Thread.currentThread().interrupt();
            }
            if (exceptionDuringRollback != null) {
                ex.addSuppressed(exceptionDuringRollback);
            }
            throw new Neo4jException(Neo4jException.GQLError.$08000.causedBy(ex).withMessage("An error occurred while closing connection"));
        }
        finally {
            this.closed = true;
            this.onClose.accept(false);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void closeBoltConnections() throws InterruptedException, ExecutionException {
        this.boltConnection.close().toCompletableFuture().get();
        Lazy<BoltConnection> lazy = this.boltConnectionForMetaData;
        synchronized (lazy) {
            if (this.boltConnectionForMetaData.isResolved()) {
                this.boltConnectionForMetaData.resolve().close();
            }
        }
    }

    @Override
    public boolean isClosed() {
        LOGGER.log(Level.FINER, () -> "Getting closed state");
        return this.closed;
    }

    @Override
    public DatabaseMetaData getMetaData() throws SQLException {
        LOGGER.log(Level.FINER, () -> "Getting metadata");
        this.assertIsOpen();
        return this.databaseMetadData.resolve().unwrap(Neo4jDatabaseMetaData.class).flush();
    }

    @Override
    public void setReadOnly(boolean readOnly) throws SQLException {
        LOGGER.log(Level.FINER, () -> "Setting read only to %s".formatted(readOnly));
        this.assertIsOpen();
        if (this.transaction != null && this.transaction.isOpen()) {
            throw new Neo4jException(Neo4jException.withReason("Updating read only setting during an unfinished transaction is not permitted"));
        }
        this.readOnly = readOnly;
    }

    @Override
    public boolean isReadOnly() throws SQLException {
        LOGGER.log(Level.FINER, () -> "Getting read only state");
        this.assertIsOpen();
        return this.readOnly;
    }

    @Override
    public void setCatalog(String catalog) throws SQLException {
        LOGGER.log(Level.FINER, () -> "Setting catalog to `%s`".formatted(catalog));
        this.assertIsOpen();
        if (this.databaseName == null || !this.databaseName.equalsIgnoreCase(catalog)) {
            throw new SQLFeatureNotSupportedException("Changing the catalog is not implemented");
        }
    }

    @Override
    public String getCatalog() throws SQLException {
        LOGGER.log(Level.FINER, () -> "Getting catalog");
        this.assertIsOpen();
        return this.getDatabaseName();
    }

    @Override
    public void setTransactionIsolation(int level) throws SQLException {
        throw new SQLFeatureNotSupportedException("Setting transaction isolation level is not supported");
    }

    @Override
    public int getTransactionIsolation() throws SQLException {
        LOGGER.log(Level.FINER, () -> "Getting transaction isolation");
        this.assertIsOpen();
        return 2;
    }

    @Override
    public SQLWarning getWarnings() throws SQLException {
        LOGGER.log(Level.FINER, () -> "Getting warnings");
        this.assertIsOpen();
        return this.warnings.get();
    }

    @Override
    public void clearWarnings() throws SQLException {
        LOGGER.log(Level.FINER, () -> "Clearing warnings");
        this.assertIsOpen();
        this.warnings.clear();
    }

    @Override
    public Statement createStatement(int resultSetType, int resultSetConcurrency) throws SQLException {
        return this.createStatement(resultSetType, resultSetConcurrency, 2);
    }

    @Override
    public PreparedStatement prepareStatement(String sql, int resultSetType, int resultSetConcurrency) throws SQLException {
        return this.prepareStatement(sql, resultSetType, resultSetConcurrency, 2);
    }

    private static void assertValidResultSetTypeAndConcurrency(int resultSetType, int resultSetConcurrency) throws SQLException {
        if (resultSetType != 1003) {
            throw new SQLFeatureNotSupportedException("Unsupported result set type: " + resultSetType);
        }
        if (resultSetConcurrency != 1007) {
            throw new SQLFeatureNotSupportedException("Unsupported result set concurrency: " + resultSetConcurrency);
        }
    }

    @Override
    public CallableStatement prepareCall(String sql, int resultSetType, int resultSetConcurrency) throws SQLException {
        return this.prepareCall(sql, resultSetType, resultSetConcurrency, 2);
    }

    @Override
    public Map<String, Class<?>> getTypeMap() {
        LOGGER.log(Level.FINER, () -> "Getting type map");
        return Map.of();
    }

    @Override
    public void setTypeMap(Map<String, Class<?>> map) throws SQLException {
        LOGGER.log(Level.FINER, () -> "Setting type map");
        Neo4jConversions.assertTypeMap(map);
    }

    @Override
    public void setHoldability(int holdability) throws SQLException {
        LOGGER.log(Level.FINER, () -> "Setting holdability to %d".formatted(holdability));
        if (holdability != 2) {
            throw new SQLFeatureNotSupportedException();
        }
    }

    @Override
    public int getHoldability() throws SQLException {
        LOGGER.log(Level.FINER, () -> "Getting holdability");
        this.assertIsOpen();
        return 2;
    }

    @Override
    public Savepoint setSavepoint() throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public Savepoint setSavepoint(String name) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public void rollback(Savepoint savepoint) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public void releaseSavepoint(Savepoint savepoint) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public Statement createStatement(int resultSetType, int resultSetConcurrency, int resultSetHoldability) throws SQLException {
        LOGGER.log(Level.FINER, () -> "Creating statement with with type %d, concurrency %d and holdability %d".formatted(resultSetType, resultSetConcurrency, resultSetHoldability));
        this.assertIsOpen();
        ConnectionImpl.assertValidResultSetTypeAndConcurrency(resultSetType, resultSetConcurrency);
        ConnectionImpl.assertValidResultSetHoldability(resultSetHoldability);
        Warnings localWarnings = new Warnings();
        return this.trackStatement(new StatementImpl(this, this::getTransaction, this.getTranslator(localWarnings), localWarnings, this::notifyStatementListeners));
    }

    @Override
    public PreparedStatement prepareStatement(String sql, int resultSetType, int resultSetConcurrency, int resultSetHoldability) throws SQLException {
        return this.prepareStatement(sql, resultSetType, resultSetConcurrency, resultSetHoldability, 2);
    }

    private PreparedStatement prepareStatement(String sql, int resultSetType, int resultSetConcurrency, int resultSetHoldability, int autoGeneratedKeys) throws SQLException {
        LOGGER.log(Level.FINER, () -> "Preparing statement `%s` with type %d, concurrency %d and holdability %d".formatted(sql, resultSetType, resultSetConcurrency, resultSetHoldability));
        this.assertIsOpen();
        ConnectionImpl.assertValidResultSetTypeAndConcurrency(resultSetType, resultSetConcurrency);
        ConnectionImpl.assertValidResultSetHoldability(resultSetHoldability);
        StatementImpl.assertAutoGeneratedKeys(autoGeneratedKeys);
        Warnings localWarnings = new Warnings();
        return this.trackStatement(new PreparedStatementImpl(this, this::getTransaction, this.getTranslator(localWarnings), localWarnings, this::notifyStatementListeners, this.rewritePlaceholders, this.rewriteBatchedStatements, autoGeneratedKeys, sql));
    }

    @Override
    public CallableStatement prepareCall(String sql, int resultSetType, int resultSetConcurrency, int resultSetHoldability) throws SQLException {
        LOGGER.log(Level.FINER, () -> "Preparing call `%s` with type %d, concurrency %d and holdability %d".formatted(sql, resultSetType, resultSetConcurrency, resultSetHoldability));
        this.assertIsOpen();
        ConnectionImpl.assertValidResultSetTypeAndConcurrency(resultSetType, resultSetConcurrency);
        ConnectionImpl.assertValidResultSetHoldability(resultSetHoldability);
        return this.trackStatement(CallableStatementImpl.prepareCall(this, this::getTransaction, this::notifyStatementListeners, this.rewriteBatchedStatements, sql));
    }

    private static void assertValidResultSetHoldability(int resultSetHoldability) throws SQLException {
        if (resultSetHoldability != 2) {
            throw new SQLFeatureNotSupportedException("Unsupported result set holdability, result sets will always be closed when the underlying transaction is closed: " + resultSetHoldability);
        }
    }

    @Override
    public PreparedStatement prepareStatement(String sql, int autoGeneratedKeys) throws SQLException {
        LOGGER.log(Level.FINER, () -> "Trying to prepare statement with auto generated keys set to %d".formatted(autoGeneratedKeys));
        return this.prepareStatement(sql, 1003, 1007, 2, autoGeneratedKeys);
    }

    @Override
    public PreparedStatement prepareStatement(String sql, int[] columnIndexes) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public PreparedStatement prepareStatement(String sql, String[] columnNames) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public Clob createClob() throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public Blob createBlob() throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public NClob createNClob() throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public SQLXML createSQLXML() throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public boolean isValid(int timeout) throws SQLException {
        LOGGER.log(Level.FINER, () -> "Checking validity with timeout %d".formatted(timeout));
        if (timeout < 0) {
            throw new Neo4jException(Neo4jException.GQLError.$22N02.withTemplatedMessage("timeout", timeout));
        }
        if (this.closed || this.fatalException != null) {
            return false;
        }
        if (this.transaction != null && this.transaction.isRunnable()) {
            try {
                this.transaction.runAndDiscard("RETURN 1", Collections.emptyMap(), timeout, false);
                return true;
            }
            catch (SQLException ignored2) {
                return false;
            }
        }
        try {
            BasicResponseHandler handler = new BasicResponseHandler();
            CompletableFuture future = this.boltConnection.writeAndFlush((ResponseHandler)handler, Messages.reset(), (ImmutableObservation)BoltConnectionObservations.NoopObservation.INSTANCE).thenCompose(ignored -> handler.summaries()).toCompletableFuture();
            if (timeout > 0) {
                future.get(timeout, TimeUnit.SECONDS);
            } else {
                future.get();
            }
            return true;
        }
        catch (TimeoutException ignored3) {
            return false;
        }
        catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            throw new Neo4jException(Neo4jException.withInternal(ex, "The thread has been interrupted."));
        }
        catch (ExecutionException ex) {
            Throwable cause = Optional.ofNullable(ex.getCause()).orElse(ex);
            if (!(cause instanceof BoltFailureException)) {
                this.fatalException = new Neo4jException(Neo4jException.GQLError.$08000.withMessage("The connection is no longer valid"));
                this.handleFatalException(this.fatalException, new Neo4jException(Neo4jException.withCause(cause)));
            }
            return false;
        }
    }

    @Override
    public void setClientInfo(String name, String value) throws SQLClientInfoException {
        LOGGER.log(Level.FINER, () -> "Setting client info `%s` to `%s`".formatted(name, value));
        if (this.closed) {
            throw new SQLClientInfoException("The connection is closed", Collections.emptyMap());
        }
        try {
            this.setClientInfo0(name, value);
        }
        catch (SQLWarning ex) {
            this.warnings.accept(ex);
        }
    }

    private void setClientInfo0(String name, String value) throws SQLWarning {
        if (name == null || name.isBlank()) {
            SQLClientInfoException throwable = new SQLClientInfoException(Map.of("", ClientInfoStatus.REASON_UNKNOWN));
            throw new SQLWarning("Client information without a name are not supported", throwable);
        }
        if (!DatabaseMetadataImpl.isSupportedClientInfoProperty(name)) {
            SQLClientInfoException throwable = new SQLClientInfoException(Map.of(name, ClientInfoStatus.REASON_UNKNOWN_PROPERTY));
            throw new SQLWarning("Unknown client info property `" + name + "`", throwable);
        }
        if (value == null || value.isBlank()) {
            SQLClientInfoException throwable = new SQLClientInfoException(Map.of("", ClientInfoStatus.REASON_VALUE_INVALID));
            throw new SQLWarning("Client information without a value are not supported", throwable);
        }
        this.clientInfo.put(name, value);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void setClientInfo(Properties properties) throws SQLClientInfoException {
        LOGGER.log(Level.FINER, () -> "Setting client info via properties");
        if (this.closed) {
            throw new SQLClientInfoException("The connection is closed", Collections.emptyMap());
        }
        Objects.requireNonNull(properties);
        HashMap<String, ClientInfoStatus> failedProperties = new HashMap<String, ClientInfoStatus>();
        ConnectionImpl connectionImpl = this;
        synchronized (connectionImpl) {
            for (String key : properties.stringPropertyNames()) {
                try {
                    this.setClientInfo0(key, properties.getProperty(key));
                }
                catch (SQLWarning ex) {
                    failedProperties.putAll(((SQLClientInfoException)ex.getCause()).getFailedProperties());
                }
            }
        }
        if (!failedProperties.isEmpty()) {
            SQLClientInfoException throwable = new SQLClientInfoException(Collections.unmodifiableMap(failedProperties));
            this.warnings.accept(new SQLWarning("There have been issues setting some properties", throwable));
        }
    }

    @Override
    public String getClientInfo(String name) throws SQLException {
        LOGGER.log(Level.FINER, () -> "Getting client info `%s`".formatted(name));
        this.assertIsOpen();
        return this.clientInfo.get(name);
    }

    @Override
    public Properties getClientInfo() throws SQLException {
        LOGGER.log(Level.FINER, () -> "Getting client info");
        this.assertIsOpen();
        Properties result = new Properties();
        result.putAll(this.clientInfo);
        return result;
    }

    @Override
    public Array createArrayOf(String typeName, Object[] elements) throws SQLException {
        return ArrayImpl.of(this, typeName, elements);
    }

    @Override
    public Struct createStruct(String typeName, Object[] attributes) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public void setSchema(String schema) throws SQLException {
        this.assertIsOpen();
    }

    @Override
    public String getSchema() throws SQLException {
        LOGGER.log(Level.FINER, () -> "Getting schema");
        this.assertIsOpen();
        return "public";
    }

    @Override
    public void abort(Executor ignored) throws SQLException {
        LOGGER.log(Level.FINER, () -> "Trying to abort the current transaction");
        if (this.closed) {
            return;
        }
        if (this.fatalException != null) {
            this.closed = true;
            return;
        }
        this.fatalException = new Neo4jException(Neo4jException.withReason("The connection has been explicitly aborted."));
        if (this.transaction != null && this.transaction.isRunnable()) {
            this.transaction.fail(this.fatalException);
        }
        try {
            this.closeBoltConnections();
        }
        catch (InterruptedException | ExecutionException ex) {
            if (ex instanceof InterruptedException) {
                Thread.currentThread().interrupt();
            }
            this.fatalException.addSuppressed(ex);
        }
        finally {
            this.closed = true;
            this.onClose.accept(true);
        }
    }

    @Override
    public void setNetworkTimeout(Executor executor, int milliseconds) throws SQLException {
        LOGGER.log(Level.FINER, () -> "Setting network timeout to %d milliseconds".formatted(milliseconds));
        this.assertIsOpen();
        if (milliseconds < 0) {
            throw new Neo4jException(Neo4jException.GQLError.$22N02.withTemplatedMessage("network timeout", milliseconds));
        }
        this.networkTimeout = milliseconds;
        if (milliseconds == 0) {
            this.boltConnection.defaultReadTimeout().ifPresent(defaultTimeout -> LOGGER.log(Level.FINE, String.format("setNetworkTimeout has been called with 0, will use the Bolt server default of % d milliseconds.", defaultTimeout.toMillis())));
            this.setReadTimeout0(null);
        } else {
            this.setReadTimeout0(Duration.ofMillis(this.networkTimeout));
        }
    }

    private void setReadTimeout0(Duration duration) throws Neo4jException {
        String failureMessage = "Failed to set read timeout";
        try {
            this.boltConnection.setReadTimeout(duration).toCompletableFuture().get();
        }
        catch (ExecutionException ex) {
            throw new Neo4jException(Neo4jException.withInternal(ex, failureMessage));
        }
        catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            throw new Neo4jException(Neo4jException.withInternal(ex, failureMessage));
        }
    }

    @Override
    public int getNetworkTimeout() {
        LOGGER.log(Level.FINER, () -> "Getting network timeout");
        return this.networkTimeout;
    }

    @Override
    public <T> T unwrap(Class<T> iface) throws SQLException {
        LOGGER.log(Level.FINER, () -> "Unwrapping `%s` into `%s`".formatted(this.getClass().getCanonicalName(), iface.getCanonicalName()));
        if (iface.isAssignableFrom(this.getClass())) {
            return iface.cast(this);
        }
        throw new Neo4jException(Neo4jException.withReason("This object does not implement the given interface"));
    }

    @Override
    public boolean isWrapperFor(Class<?> iface) throws SQLException {
        return iface.isAssignableFrom(this.getClass());
    }

    private void assertIsOpen() throws SQLException {
        if (this.closed) {
            throw new Neo4jException(Neo4jException.GQLError.$08000.withMessage("The connection is closed"));
        }
    }

    Neo4jTransaction getTransaction(Map<String, Object> additionalTransactionMetadata) throws SQLException {
        this.assertIsOpen();
        if (this.fatalException != null) {
            throw this.fatalException;
        }
        if (this.transaction != null && this.transaction.isOpen()) {
            if (this.transaction.isAutoCommit()) {
                throw new SQLFeatureNotSupportedException("Only a single autocommit transaction is supported");
            }
            return this.transaction;
        }
        Map<String, Object> combinedTransactionMetadata = this.getCombinedTransactionMetadata(additionalTransactionMetadata);
        this.transaction = new DefaultTransactionImpl(this.boltConnection, this.bookmarkManager, combinedTransactionMetadata, this::handleFatalException, this.resetNeeded.getAndSet(false), this.autoCommit, this.getAccessMode(), null, this.databaseName, state -> this.resetNeeded.compareAndSet(false, EnumSet.of(Neo4jTransaction.State.FAILED, Neo4jTransaction.State.OPEN_FAILED).contains(state)), this.authenticationManager.getOrRefresh());
        return this.transaction;
    }

    Neo4jTransaction newMetadataTransaction(Map<String, Object> additionalTransactionMetadata) throws SQLException {
        Map<String, Object> combinedTransactionMetadata = this.getCombinedTransactionMetadata(additionalTransactionMetadata);
        return new DefaultTransactionImpl(this.boltConnectionForMetaData.resolve(), this.bookmarkManager, combinedTransactionMetadata, this::handleFatalException, false, this.autoCommit, this.getAccessMode(), null, this.databaseName, state -> {}, this.authenticationManager.getOrRefresh());
    }

    private Map<String, Object> getCombinedTransactionMetadata(Map<String, Object> additionalTransactionMetadata) throws SQLException {
        HashMap<String, Object> combinedTransactionMetadata = new HashMap<String, Object>(this.transactionMetadata.size() + additionalTransactionMetadata.size() + 1);
        combinedTransactionMetadata.putAll(this.transactionMetadata);
        combinedTransactionMetadata.putAll(additionalTransactionMetadata);
        if (!combinedTransactionMetadata.containsKey("app")) {
            combinedTransactionMetadata.put("app", this.getApp());
        }
        return combinedTransactionMetadata;
    }

    private AccessMode getAccessMode() {
        return this.readOnly ? AccessMode.READ : AccessMode.WRITE;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void flushTranslationCache() {
        LOGGER.log(Level.FINER, () -> "Flushing translation cache");
        ConnectionImpl connectionImpl = this;
        synchronized (connectionImpl) {
            this.l2cache.flush();
            this.translators.resolve().forEach(Translator::flushCache);
        }
    }

    @Override
    public String getDatabaseName() {
        LOGGER.log(Level.FINER, () -> "Getting database name");
        return this.databaseName;
    }

    @Override
    public void addListener(ConnectionListener connectionListener) {
        this.listeners.add(Objects.requireNonNull(connectionListener));
        this.authenticationManager.addListener(connectionListener);
    }

    private <T extends StatementImpl> T trackStatement(T statement) {
        this.purgeClearedStatementReferences();
        this.trackedStatementReferences.add(new WeakReference<Statement>(statement, this.trackedStatementReferenceQueue));
        if (!this.listeners.isEmpty()) {
            this.listeners.forEach(listener -> {
                if (listener instanceof StatementListener) {
                    StatementListener statementListener = (StatementListener)((Object)listener);
                    statement.addListener(statementListener);
                }
            });
            Class<? extends Statement> type = statement.getType();
            ConnectionListener.StatementCreatedEvent statementCreatedEvent = new ConnectionListener.StatementCreatedEvent(this.databaseUrl, type, statement);
            Events.notify(this.listeners, listener -> listener.onStatementCreated(statementCreatedEvent));
        }
        return statement;
    }

    private void purgeClearedStatementReferences() {
        Reference<Statement> reference = this.trackedStatementReferenceQueue.poll();
        while (reference != null) {
            this.trackedStatementReferences.remove(reference);
            reference = this.trackedStatementReferenceQueue.poll();
        }
    }

    private void handleFatalException(SQLException fatalSqlException, SQLException sqlException) {
        Throwable cause = sqlException.getCause();
        if (cause instanceof BoltConnectionReadTimeoutException) {
            for (Reference<Statement> reference : this.trackedStatementReferences) {
                Statement statement = reference.get();
                if (statement == null) continue;
                try {
                    statement.close();
                }
                catch (Exception ex) {
                    sqlException.addSuppressed(ex);
                }
            }
            try {
                this.close();
            }
            catch (SQLException ex) {
                sqlException.addSuppressed(ex);
            }
        } else {
            this.fatalException = fatalSqlException;
        }
    }

    @Override
    public Neo4jConnection withMetadata(Map<String, Object> metadata) {
        LOGGER.log(Level.FINER, () -> "Adding new transaction metadata");
        if (metadata != null) {
            this.transactionMetadata.putAll(metadata);
        }
        return this;
    }

    String getApp() throws SQLException {
        String applicationName = this.getClientInfo("ApplicationName");
        return String.format("%sJava/%s (%s %s %s) neo4j-jdbc/%s", applicationName == null || applicationName.isBlank() ? "" : applicationName.trim() + " ", Optional.ofNullable(System.getProperty("java.version")).filter(Predicate.not(String::isBlank)).orElse("-"), Optional.ofNullable(System.getProperty("java.vm.vendor")).filter(Predicate.not(String::isBlank)).orElse("-"), Optional.ofNullable(System.getProperty("java.vm.name")).filter(Predicate.not(String::isBlank)).orElse("-"), Optional.ofNullable(System.getProperty("java.vm.version")).filter(Predicate.not(String::isBlank)).orElse("-"), Optional.ofNullable(System.getProperty("neo4j.jdbc.version")).filter(Predicate.not(String::isBlank)).orElseGet(ProductVersion::getValue));
    }

    @Override
    public URI getDatabaseURL() {
        return this.databaseUrl;
    }

    @Override
    public Neo4jConnection withTracer(Neo4jTracer tracer) {
        this.setTracer(tracer);
        return this;
    }

    @Override
    public void setTracer(Neo4jTracer tracer) {
        if (tracer != null && this.listeners.stream().noneMatch(l -> {
            Tracing t;
            return l instanceof Tracing && (t = (Tracing)l).usingSameTracer(tracer);
        })) {
            this.addListener(new Tracing(tracer, this));
        }
    }

    static boolean forceCypher(String sql) {
        Matcher matcher = PATTERN_ENFORCE_CYPHER.matcher(sql);
        while (matcher.find()) {
            if (matcher.group(1) != null && matcher.group(1).equals(matcher.group(2))) continue;
            return true;
        }
        return false;
    }

    static class TranslatorChain
    implements UnaryOperator<String> {
        private final List<Translator> translators;
        private final DatabaseMetaData metaData;
        private final Consumer<SQLWarning> warningSink;

        TranslatorChain(List<Translator> translators, DatabaseMetaData metaData, Consumer<SQLWarning> warningSink) {
            this.translators = translators;
            this.metaData = metaData;
            this.warningSink = warningSink;
        }

        @Override
        public String apply(String statement) {
            Throwable lastException = null;
            String result = null;
            String in = statement;
            for (Translator translator : this.translators) {
                if (ConnectionImpl.forceCypher(in)) {
                    result = in;
                    break;
                }
                try {
                    result = translator.translate(in, this.metaData);
                    if (result == null) continue;
                    in = result;
                }
                catch (IllegalArgumentException ex) {
                    if (this.translators.size() == 1) {
                        throw ex;
                    }
                    this.warningSink.accept(new SQLWarning("Translator %s failed to translate `%s`".formatted(translator.getClass().getName(), in), ex));
                    if (ex.getCause() != null) {
                        lastException = ex.getCause();
                        continue;
                    }
                    lastException = ex;
                }
            }
            if (result == null) {
                throw new IllegalStateException("No suitable translator for input `%s`".formatted(statement), lastException);
            }
            return result;
        }
    }
}

