/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.pki;

import java.io.IOException;
import java.math.BigInteger;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.security.AlgorithmParameters;
import java.security.Key;
import java.security.KeyException;
import java.security.KeyFactory;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.spec.DSAPrivateKeySpec;
import java.security.spec.ECParameterSpec;
import java.security.spec.ECPrivateKeySpec;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.InvalidParameterSpecException;
import java.security.spec.KeySpec;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.RSAPrivateCrtKeySpec;
import java.security.spec.RSAPublicKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.HexFormat;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.function.Function;
import javax.crypto.Cipher;
import javax.crypto.EncryptedPrivateKeyInfo;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.SecretKey;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.PBEKeySpec;
import javax.crypto.spec.SecretKeySpec;
import org.neo4j.pki.DerUtils;

final class PemFormats {
    private static final KeyFactory RSA_KEY_FACTORY;
    private static final KeyFactory DSA_KEY_FACTORY;
    private static final KeyFactory EC_KEY_FACTORY;
    private static final Function<PKCS8EncodedKeySpec, PrivateKey> ALL_KEY_FACTORIES;

    private PemFormats() {
    }

    private static void assertNoPassword(String password) throws KeyException {
        if (password != null) {
            throw new KeyException("Passphrase was provided but found un-encrypted private key.");
        }
    }

    private static void assertPassword(String password) throws KeyException {
        if (password == null) {
            throw new KeyException("Found encrypted private key but no passphrase was provided.");
        }
    }

    private static byte[] decryptLegacyPem(byte[] der, String algorithm, byte[] iv, String password) throws KeyException {
        try {
            DecryptSchema decryptSchema;
            try {
                decryptSchema = DecryptSchema.valueOf(algorithm.replace("-", "_"));
            }
            catch (IllegalArgumentException e) {
                throw new KeyException(String.format("Encryption scheme %s is not supported.", algorithm));
            }
            byte[] kdf = PemFormats.keyDerivationFunction(iv, password);
            byte[] key = new byte[decryptSchema.keySize];
            System.arraycopy(kdf, 0, key, 0, decryptSchema.keySize);
            SecretKeySpec secret = new SecretKeySpec(key, decryptSchema.family);
            Cipher cipher = Cipher.getInstance(decryptSchema.cipher);
            cipher.init(2, (Key)secret, new IvParameterSpec(iv));
            return cipher.doFinal(der);
        }
        catch (Exception e) {
            throw new KeyException("Failed to decrypt PEM file.", e);
        }
    }

    private static byte[] keyDerivationFunction(byte[] iv, String password) throws NoSuchAlgorithmException {
        byte[] pw = password.getBytes(StandardCharsets.UTF_8);
        MessageDigest md5 = MessageDigest.getInstance("MD5");
        md5.update(pw);
        md5.update(iv, 0, 8);
        byte[] d0 = md5.digest();
        md5.update(d0);
        md5.update(pw);
        md5.update(iv, 0, 8);
        byte[] d1 = md5.digest();
        byte[] kdf = new byte[32];
        System.arraycopy(d0, 0, kdf, 0, 16);
        System.arraycopy(d1, 0, kdf, 16, 16);
        return kdf;
    }

    static {
        try {
            RSA_KEY_FACTORY = KeyFactory.getInstance("RSA");
            DSA_KEY_FACTORY = KeyFactory.getInstance("DSA");
            EC_KEY_FACTORY = KeyFactory.getInstance("EC");
            ALL_KEY_FACTORIES = keySpec -> {
                try {
                    return RSA_KEY_FACTORY.generatePrivate((KeySpec)keySpec);
                }
                catch (InvalidKeySpecException e) {
                    try {
                        return DSA_KEY_FACTORY.generatePrivate((KeySpec)keySpec);
                    }
                    catch (InvalidKeySpecException ex) {
                        try {
                            return EC_KEY_FACTORY.generatePrivate((KeySpec)keySpec);
                        }
                        catch (InvalidKeySpecException exc) {
                            e.addSuppressed(ex);
                            e.addSuppressed(exc);
                            throw new IllegalStateException("Key does not match RSA, DSA or EC spec.", e);
                        }
                    }
                }
            };
        }
        catch (NoSuchAlgorithmException e) {
            throw new IllegalStateException("Non-conforming JDK implementation.", e);
        }
    }

    private static enum DecryptSchema {
        DES_CBC(8, "DES", "DES/CBC/PKCS5Padding"),
        DES_EDE3_CBC(24, "DESede", "DESede/CBC/PKCS5Padding"),
        AES_128_CBC(16, "AES", "AES/CBC/PKCS5Padding"),
        AES_192_CBC(24, "AES", "AES/CBC/PKCS5Padding"),
        AES_256_CBC(32, "AES", "AES/CBC/PKCS5Padding");

        final int keySize;
        final String family;
        final String cipher;

        private DecryptSchema(int keySize, String family, String cipher) {
            this.keySize = keySize;
            this.family = family;
            this.cipher = cipher;
        }
    }

    static class PemPKCS1Ec
    extends PemLegacy {
        static final String PRIVATE_LABEL = "EC PRIVATE KEY";

        PemPKCS1Ec() {
        }

        @Override
        public PublicKey decodePublicKey(byte[] der) {
            throw new UnsupportedOperationException();
        }

        @Override
        protected PrivateKey decodePrivate0(ByteBuffer buffer) throws InvalidKeySpecException {
            try {
                BigInteger s = new BigInteger(1, DerUtils.readDerOctetString(buffer));
                byte[] parameters = DerUtils.getDerContext(buffer, (byte)0);
                AlgorithmParameters ecParams = AlgorithmParameters.getInstance("EC");
                ecParams.init(parameters);
                ECParameterSpec parameterSpec = ecParams.getParameterSpec(ECParameterSpec.class);
                return EC_KEY_FACTORY.generatePrivate(new ECPrivateKeySpec(s, parameterSpec));
            }
            catch (IOException | NoSuchAlgorithmException | InvalidParameterSpecException e) {
                throw new IllegalArgumentException("Failed to decode EC private key", e);
            }
        }

        @Override
        protected BigInteger version() {
            return BigInteger.ONE;
        }
    }

    static class PemPKCS1Dsa
    extends PemLegacy {
        static final String PRIVATE_LABEL = "DSA PRIVATE KEY";

        PemPKCS1Dsa() {
        }

        @Override
        public PublicKey decodePublicKey(byte[] der) {
            throw new UnsupportedOperationException();
        }

        @Override
        protected PrivateKey decodePrivate0(ByteBuffer buffer) throws InvalidKeySpecException {
            BigInteger p = DerUtils.readDerInteger(buffer);
            BigInteger q = DerUtils.readDerInteger(buffer);
            BigInteger g = DerUtils.readDerInteger(buffer);
            DerUtils.readDerInteger(buffer);
            BigInteger x = DerUtils.readDerInteger(buffer);
            return DSA_KEY_FACTORY.generatePrivate(new DSAPrivateKeySpec(x, p, q, g));
        }

        @Override
        protected BigInteger version() {
            return BigInteger.ZERO;
        }
    }

    static class PemPKCS1Rsa
    extends PemLegacy {
        static final String PRIVATE_LABEL = "RSA PRIVATE KEY";
        static final String PUBLIC_LABEL = "RSA PUBLIC KEY";

        PemPKCS1Rsa() {
        }

        @Override
        public PublicKey decodePublicKey(byte[] der) throws KeyException {
            ByteBuffer input = ByteBuffer.wrap(der);
            if (DerUtils.beginDerSequence(input) != input.remaining()) {
                throw new IllegalArgumentException("Malformed RSAPublicKey");
            }
            BigInteger n = DerUtils.readDerInteger(input);
            BigInteger e = DerUtils.readDerInteger(input);
            try {
                return RSA_KEY_FACTORY.generatePublic(new RSAPublicKeySpec(n, e));
            }
            catch (InvalidKeySpecException ex) {
                throw new KeyException(ex);
            }
        }

        @Override
        protected PrivateKey decodePrivate0(ByteBuffer buffer) throws InvalidKeySpecException {
            BigInteger n = DerUtils.readDerInteger(buffer);
            BigInteger e = DerUtils.readDerInteger(buffer);
            BigInteger d = DerUtils.readDerInteger(buffer);
            BigInteger p = DerUtils.readDerInteger(buffer);
            BigInteger q = DerUtils.readDerInteger(buffer);
            BigInteger ep = DerUtils.readDerInteger(buffer);
            BigInteger eq = DerUtils.readDerInteger(buffer);
            BigInteger c = DerUtils.readDerInteger(buffer);
            return RSA_KEY_FACTORY.generatePrivate(new RSAPrivateCrtKeySpec(n, e, d, p, q, ep, eq, c));
        }

        @Override
        protected BigInteger version() {
            return BigInteger.ZERO;
        }
    }

    static abstract class PemLegacy
    implements PemFormat {
        PemLegacy() {
        }

        @Override
        public PrivateKey decodePrivate(byte[] der, Map<String, String> headers, String password) throws KeyException {
            String procType = headers.get("Proc-Type");
            if (procType != null && procType.equals("4,ENCRYPTED")) {
                PemFormats.assertPassword(password);
                String deckInfo = headers.get("DEK-Info");
                if (deckInfo == null) {
                    throw new KeyException("Missing 'DEK-Info' in encrypted PRIVATE KEY.");
                }
                StringTokenizer tokenizer = new StringTokenizer(deckInfo, ",");
                String algorithm = tokenizer.nextToken();
                byte[] iv = HexFormat.of().parseHex(tokenizer.nextToken());
                der = PemFormats.decryptLegacyPem(der, algorithm, iv, password);
            } else {
                PemFormats.assertNoPassword(password);
            }
            ByteBuffer buffer = ByteBuffer.wrap(der);
            if (DerUtils.beginDerSequence(buffer) != buffer.remaining()) {
                throw new IllegalArgumentException("Malformed ASN.1 input.");
            }
            if (!this.version().equals(DerUtils.readDerInteger(buffer))) {
                throw new IllegalArgumentException("PrivateKey version mismatch.");
            }
            try {
                return this.decodePrivate0(buffer);
            }
            catch (InvalidKeySpecException e) {
                throw new KeyException(e);
            }
        }

        protected abstract PrivateKey decodePrivate0(ByteBuffer var1) throws InvalidKeySpecException;

        protected abstract BigInteger version();
    }

    static class Pkcs8Encrypted
    extends Pkcs8 {
        static final String ENCRYPTED_LABEL = "ENCRYPTED PRIVATE KEY";

        Pkcs8Encrypted() {
        }

        @Override
        public PrivateKey decodePrivate(byte[] der, Map<String, String> headers, String password) throws KeyException {
            PemFormats.assertPassword(password);
            try {
                EncryptedPrivateKeyInfo keyInfo = new EncryptedPrivateKeyInfo(der);
                SecretKey pbeKey = Pkcs8Encrypted.getSecretKey(keyInfo, password);
                Cipher cipher = Pkcs8Encrypted.getCipher(keyInfo);
                cipher.init(2, (Key)pbeKey, keyInfo.getAlgParameters());
                return ALL_KEY_FACTORIES.apply(keyInfo.getKeySpec(cipher));
            }
            catch (Exception e) {
                throw new KeyException("Unable to decrypt private key.", e);
            }
        }

        private static SecretKey getSecretKey(EncryptedPrivateKeyInfo keyInfo, String password) throws InvalidKeySpecException, NoSuchAlgorithmException {
            SecretKeyFactory keyFactory;
            try {
                keyFactory = SecretKeyFactory.getInstance(keyInfo.getAlgName());
            }
            catch (NoSuchAlgorithmException e) {
                keyFactory = SecretKeyFactory.getInstance(keyInfo.getAlgParameters().toString());
            }
            return keyFactory.generateSecret(new PBEKeySpec(password.toCharArray()));
        }

        private static Cipher getCipher(EncryptedPrivateKeyInfo keyInfo) throws NoSuchPaddingException, NoSuchAlgorithmException {
            try {
                return Cipher.getInstance(keyInfo.getAlgName());
            }
            catch (NoSuchAlgorithmException | NoSuchPaddingException e) {
                return Cipher.getInstance(keyInfo.getAlgParameters().toString());
            }
        }
    }

    static class Pkcs8
    implements PemFormat {
        static final String PRIVATE_LABEL = "PRIVATE KEY";
        static final String PUBLIC_LABEL = "PUBLIC KEY";

        Pkcs8() {
        }

        @Override
        public PrivateKey decodePrivate(byte[] der, Map<String, String> headers, String password) throws KeyException {
            PemFormats.assertNoPassword(password);
            return ALL_KEY_FACTORIES.apply(new PKCS8EncodedKeySpec(der));
        }

        @Override
        public PublicKey decodePublicKey(byte[] der) throws KeyException {
            X509EncodedKeySpec encodedKeySpec = new X509EncodedKeySpec(der);
            try {
                return RSA_KEY_FACTORY.generatePublic(encodedKeySpec);
            }
            catch (InvalidKeySpecException e) {
                try {
                    return DSA_KEY_FACTORY.generatePublic(encodedKeySpec);
                }
                catch (InvalidKeySpecException ex) {
                    try {
                        return EC_KEY_FACTORY.generatePublic(encodedKeySpec);
                    }
                    catch (InvalidKeySpecException exc) {
                        e.addSuppressed(ex);
                        e.addSuppressed(exc);
                        throw new KeyException("Public key does not match RSA, DSA or EC spec.", e);
                    }
                }
            }
        }
    }

    static interface PemFormat {
        public PrivateKey decodePrivate(byte[] var1, Map<String, String> var2, String var3) throws KeyException;

        public PublicKey decodePublicKey(byte[] var1) throws KeyException;
    }
}

