/*
 * Decompiled with CFR 0.152.
 */
package apoc.periodic;

import apoc.Pools;
import apoc.periodic.BatchAndTotalResult;
import apoc.periodic.BatchMode;
import apoc.periodic.PeriodicUtils;
import apoc.util.Util;
import apoc.util.collection.Iterables;
import apoc.util.collection.Iterators;
import java.util.Collections;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.regex.Pattern;
import java.util.stream.Stream;
import org.apache.commons.lang3.tuple.Pair;
import org.neo4j.graphdb.GraphDatabaseService;
import org.neo4j.graphdb.QueryExecutionType;
import org.neo4j.graphdb.Result;
import org.neo4j.graphdb.Transaction;
import org.neo4j.graphdb.schema.ConstraintDefinition;
import org.neo4j.graphdb.schema.IndexDefinition;
import org.neo4j.graphdb.schema.Schema;
import org.neo4j.internal.kernel.api.procs.ProcedureCallContext;
import org.neo4j.kernel.api.QueryLanguage;
import org.neo4j.kernel.api.procedure.QueryLanguageScope;
import org.neo4j.kernel.impl.coreapi.InternalTransaction;
import org.neo4j.logging.Log;
import org.neo4j.procedure.Admin;
import org.neo4j.procedure.Context;
import org.neo4j.procedure.Description;
import org.neo4j.procedure.Mode;
import org.neo4j.procedure.Name;
import org.neo4j.procedure.Procedure;
import org.neo4j.procedure.TerminationGuard;

public class Periodic {
    static final Pattern LIMIT_PATTERN = Pattern.compile("\\slimit\\s", 2);
    @Context
    public GraphDatabaseService db;
    @Context
    public TerminationGuard terminationGuard;
    @Context
    public Log log;
    @Context
    public Pools pools;
    @Context
    public Transaction tx;
    @Context
    public ProcedureCallContext procedureCallContext;

    @Admin
    @Procedure(name="apoc.periodic.truncate", mode=Mode.SCHEMA)
    @Description(value="Removes all entities (and optionally indexes and constraints) from the database using the `apoc.periodic.iterate` procedure.")
    public void truncate(@Name(value="config", defaultValue="{}", description="{\n    dropSchema = true :: BOOLEAN,\n    batchSize = 10000 :: INTEGER,\n    parallel = false :: BOOLEAN,\n    retries = 0 :: INTEGER,\n    batchMode = \"BATCH\" :: STRING,\n    params = {} :: MAP,\n    concurrency :: INTEGER,\n    failedParams = -1 :: INTEGER,\n    planner = \"DEFAULT\" :: [\"DEFAULT\", \"COST\", \"IDP\", \"DP\"]\n}\n") Map<String, Object> config) {
        this.iterate("MATCH ()-[r]->() RETURN id(r) as id", "MATCH ()-[r]->() WHERE id(r) = id DELETE r", config);
        this.iterate("MATCH (n) RETURN id(n) as id", "MATCH (n) WHERE id(n) = id DELETE n", config);
        if (Util.toBoolean(config.get("dropSchema"))) {
            Schema schema = this.tx.schema();
            schema.getConstraints().forEach(ConstraintDefinition::drop);
            Util.getIndexes(this.tx).forEach(IndexDefinition::drop);
        }
    }

    @Procedure(value="apoc.periodic.list")
    @Description(value="Returns a `LIST<ANY>` of all background jobs.")
    public Stream<PeriodicUtils.JobInfo> list() {
        return this.pools.getJobList().entrySet().stream().map(e -> ((PeriodicUtils.JobInfo)e.getKey()).update((Future)e.getValue()));
    }

    @Procedure(name="apoc.periodic.commit", mode=Mode.WRITE)
    @Description(value="Runs the given statement in separate batched transactions.")
    public Stream<RundownResult> commit(@Name(value="statement", description="The Cypher statement to run.") String statement, @Name(value="params", defaultValue="{}", description="The parameters for the given Cypher statement.") Map<String, Object> parameters) {
        this.validateQuery(statement);
        Map<Object, Object> params = parameters == null ? Collections.emptyMap() : parameters;
        long total = 0L;
        long executions = 0L;
        long updates = 0L;
        long start = System.nanoTime();
        if (!LIMIT_PATTERN.matcher(statement).find()) {
            throw new IllegalArgumentException("the statement sent to apoc.periodic.commit must contain a `limit`");
        }
        AtomicInteger batches = new AtomicInteger();
        AtomicInteger failedCommits = new AtomicInteger();
        ConcurrentHashMap<String, Long> commitErrors = new ConcurrentHashMap<String, Long>();
        AtomicInteger failedBatches = new AtomicInteger();
        ConcurrentHashMap<String, Long> batchErrors = new ConcurrentHashMap<String, Long>();
        String periodicId = UUID.randomUUID().toString();
        if (this.log.isDebugEnabled()) {
            this.log.debug("Starting periodic commit from `%s` in separate thread with id: `%s`", new Object[]{statement, periodicId});
        }
        do {
            Map<String, Object> window = Util.map("_count", updates, "_total", total);
            updates = Util.getFuture(this.pools.getScheduledExecutorService().submit(() -> {
                batches.incrementAndGet();
                try {
                    return this.executeNumericResultStatement(statement, Util.merge(window, params));
                }
                catch (Exception e) {
                    failedBatches.incrementAndGet();
                    PeriodicUtils.recordError(batchErrors, e);
                    return 0L;
                }
            }), commitErrors, failedCommits, 0L);
            total += updates;
            if (updates > 0L) {
                ++executions;
            }
            if (!this.log.isDebugEnabled()) continue;
            this.log.debug("Processed in periodic commit with id %s, no %d executions", new Object[]{periodicId, executions});
        } while (updates > 0L && !Util.transactionIsTerminated(this.terminationGuard));
        if (this.log.isDebugEnabled()) {
            this.log.debug("Terminated periodic commit with id %s with %d executions", new Object[]{periodicId, executions});
        }
        long timeTaken = TimeUnit.NANOSECONDS.toSeconds(System.nanoTime() - start);
        boolean wasTerminated = Util.transactionIsTerminated(this.terminationGuard);
        return Stream.of(new RundownResult(total, executions, timeTaken, batches.get(), failedBatches.get(), batchErrors, failedCommits.get(), commitErrors, wasTerminated));
    }

    private long executeNumericResultStatement(@Name(value="statement") String statement, @Name(value="params") Map<String, Object> parameters) {
        return (Long)this.db.executeTransactionally(Util.prefixQueryWithCheck(this.procedureCallContext, statement), parameters, result -> {
            String column = (String)Iterables.single(result.columns());
            return result.columnAs(column).stream().mapToLong(o -> (Long)o).sum();
        });
    }

    @Procedure(value="apoc.periodic.cancel")
    @Description(value="Cancels the given background job.")
    public Stream<PeriodicUtils.JobInfo> cancel(@Name(value="name", description="The name of the job to cancel.") String name) {
        PeriodicUtils.JobInfo info = new PeriodicUtils.JobInfo(name);
        Future future = this.pools.getJobList().remove(info);
        if (future != null) {
            future.cancel(false);
            return Stream.of(info.update(future));
        }
        return Stream.empty();
    }

    @Procedure(name="apoc.periodic.submit", mode=Mode.WRITE)
    @Description(value="Creates a background job which runs the given Cypher statement once.")
    public Stream<PeriodicUtils.JobInfo> submit(@Name(value="name", description="The name of the job.") String name, @Name(value="statement", description="The Cypher statement to run.") String statement, @Name(value="params", defaultValue="{}", description="{ params = {} :: MAP }") Map<String, Object> config) {
        this.validateQuery(statement);
        String query = Util.prefixQueryWithCheck(this.procedureCallContext, statement);
        return PeriodicUtils.submitProc(name, query, config, this.db, this.log, this.pools);
    }

    @Procedure(name="apoc.periodic.repeat", mode=Mode.WRITE)
    @Description(value="Runs a repeatedly called background job. To stop this procedure, use `apoc.periodic.cancel`.")
    @QueryLanguageScope(scope={QueryLanguage.CYPHER_5})
    public Stream<PeriodicUtils.JobInfo> repeatCypher5(@Name(value="name", description="The name of the job.") String name, @Name(value="statement", description="The Cypher statement to run.") String statement, @Name(value="rate", description="The delay in seconds to wait between each job execution.") long rate, @Name(value="config", defaultValue="{}", description="{ params = {} :: MAP }") Map<String, Object> config) {
        this.validateQuery(statement);
        Map params = config.getOrDefault("params", Collections.emptyMap());
        String query = Util.prefixQueryWithCheck(this.procedureCallContext, statement);
        PeriodicUtils.JobInfo info = this.schedule(name, () -> this.db.executeTransactionally(query, params, Util.CONSUME_VOID), 0L, rate, true);
        return Stream.of(info);
    }

    @Procedure(name="apoc.periodic.repeat", mode=Mode.WRITE)
    @Description(value="Runs a repeatedly called background job. To stop this procedure, use `apoc.periodic.cancel`.")
    @QueryLanguageScope(scope={QueryLanguage.CYPHER_25})
    public Stream<PeriodicUtils.JobInfo> repeat(@Name(value="name", description="The name of the job.") String name, @Name(value="statement", description="The Cypher statement to run.") String statement, @Name(value="rate", description="The delay in seconds to wait between each job execution.") long rate, @Name(value="config", defaultValue="{}", description="{ params = {} :: MAP, cancelOnError = true :: BOOLEAN }") Map<String, Object> config) {
        this.validateQuery(statement);
        Map params = config.getOrDefault("params", Collections.emptyMap());
        Boolean cancelOnError = (Boolean)config.getOrDefault("cancelOnError", true);
        String query = Util.prefixQueryWithCheck(this.procedureCallContext, statement);
        PeriodicUtils.JobInfo info = this.schedule(name, () -> this.db.executeTransactionally(query, params, Util.CONSUME_VOID), 0L, rate, cancelOnError);
        return Stream.of(info);
    }

    private void validateQuery(String statement) {
        Util.validateQuery(this.db, statement, Set.of(Mode.WRITE, Mode.READ, Mode.DEFAULT), QueryExecutionType.QueryType.READ_ONLY, QueryExecutionType.QueryType.WRITE, QueryExecutionType.QueryType.READ_WRITE);
    }

    @Procedure(name="apoc.periodic.countdown", mode=Mode.WRITE)
    @Description(value="Runs a repeatedly called background statement until it returns 0.")
    public Stream<PeriodicUtils.JobInfo> countdown(@Name(value="name", description="The name of the job.") String name, @Name(value="statement", description="The Cypher statement to run, returning a count on each run indicating the remaining iterations.") String statement, @Name(value="delay", description="The delay in seconds to wait between each job execution.") long delay) {
        this.validateQuery(statement);
        String query = Util.prefixQueryWithCheck(this.procedureCallContext, statement);
        PeriodicUtils.JobInfo info = PeriodicUtils.submitJob(name, new Countdown(name, query, delay, this.log), this.log, this.pools);
        info.delay = delay;
        return Stream.of(info);
    }

    public PeriodicUtils.JobInfo schedule(String name, Runnable task, long delay, long repeat, boolean cancelOnError) {
        PeriodicUtils.JobInfo info = new PeriodicUtils.JobInfo(name, delay, repeat);
        Future future = this.pools.getJobList().remove(info);
        if (future != null) {
            future.cancel(false);
        }
        ScheduledFuture<?> newFuture = this.pools.getScheduledExecutorService().scheduleWithFixedDelay(PeriodicUtils.wrapTask(name, task, this.log, cancelOnError), delay, repeat, TimeUnit.SECONDS);
        future = this.pools.getJobList().put(info, newFuture);
        if (future != null) {
            future.cancel(false);
        }
        return info;
    }

    @Procedure(name="apoc.periodic.iterate", mode=Mode.WRITE)
    @Description(value="Runs the second statement for each item returned by the first statement.\nThis procedure returns the number of batches and the total number of processed rows.")
    public Stream<BatchAndTotalResult> iterate(@Name(value="cypherIterate", description="The first Cypher statement to be run.") String cypherIterate, @Name(value="cypherAction", description="The Cypher statement to run for each item returned by the initial Cypher statement.") String cypherAction, @Name(value="config", description="{\n    batchSize = 10000 :: INTEGER,\n    parallel = false :: BOOLEAN,\n    retries = 0 :: INTEGER,\n    batchMode = \"BATCH\" :: STRING,\n    params = {} :: MAP,\n    concurrency :: INTEGER,\n    failedParams = -1 :: INTEGER,\n    planner = \"DEFAULT\" :: [\"DEFAULT\", \"COST\", \"IDP\", \"DP\"]\n}\n") Map<String, Object> config) {
        Map metaData;
        this.validateQuery(cypherIterate);
        long batchSize = Util.toLong(config.getOrDefault("batchSize", 10000));
        if (batchSize < 1L) {
            throw new IllegalArgumentException("batchSize parameter must be > 0");
        }
        int concurrency = Util.toInteger(config.getOrDefault("concurrency", Runtime.getRuntime().availableProcessors()));
        if (concurrency < 1) {
            throw new IllegalArgumentException("concurrency parameter must be > 0");
        }
        boolean parallel = Util.toBoolean(config.getOrDefault("parallel", false));
        long retries = Util.toLong(config.getOrDefault("retries", 0));
        int failedParams = Util.toInteger(config.getOrDefault("failedParams", -1));
        Transaction transaction = this.tx;
        if (transaction instanceof InternalTransaction) {
            InternalTransaction iTx = (InternalTransaction)transaction;
            metaData = iTx.kernelTransaction().getMetaData();
        } else {
            metaData = Map.of();
        }
        BatchMode batchMode = BatchMode.fromConfig(config);
        Map params = config.getOrDefault("params", Collections.emptyMap());
        try (Result result = this.tx.execute(Util.slottedRuntime(cypherIterate, Util.getCypherVersionString(this.procedureCallContext)), params);){
            Pair<String, Boolean> prepared = PeriodicUtils.prepareInnerStatement(cypherAction, batchMode, result.columns(), "_batch");
            String innerStatement = Util.applyPlanner(prepared.getLeft(), Util.Planner.valueOf((String)config.getOrDefault("planner", Util.Planner.DEFAULT.name())), Util.getCypherVersionString(this.procedureCallContext));
            boolean iterateList = prepared.getRight();
            String periodicId = UUID.randomUUID().toString();
            if (this.log.isDebugEnabled()) {
                this.log.debug("Starting periodic iterate from `%s` operation using iteration `%s` in separate thread with id: `%s`", new Object[]{cypherIterate, cypherAction, periodicId});
            }
            Stream<BatchAndTotalResult> stream = PeriodicUtils.iterateAndExecuteBatchedInSeparateThread(this.db, this.terminationGuard, this.log, this.pools, (int)batchSize, parallel, iterateList, retries, (Iterator<Map<String, Object>>)result, (tx, p) -> {
                if (tx instanceof InternalTransaction) {
                    InternalTransaction iTx = (InternalTransaction)tx;
                    iTx.setMetaData(metaData);
                }
                Result r = tx.execute(innerStatement, Util.merge(params, p));
                Iterators.count(r);
                return r.getQueryStatistics();
            }, concurrency, failedParams, periodicId);
            return stream;
        }
    }

    public static class RundownResult {
        @Description(value="The total number of updates.")
        public final long updates;
        @Description(value="The total number of executions.")
        public final long executions;
        @Description(value="The total time taken in nanoseconds.")
        public final long runtime;
        @Description(value="The number of run batches.")
        public final long batches;
        @Description(value="The number of failed batches.")
        public final long failedBatches;
        @Description(value="Errors returned from the failed batches.")
        public final Map<String, Long> batchErrors;
        @Description(value="The number of failed commits.")
        public final long failedCommits;
        @Description(value="Errors returned from the failed commits.")
        public final Map<String, Long> commitErrors;
        @Description(value="If the job was terminated.")
        public final boolean wasTerminated;

        public RundownResult(long total, long executions, long timeTaken, long batches, long failedBatches, Map<String, Long> batchErrors, long failedCommits, Map<String, Long> commitErrors, boolean wasTerminated) {
            this.updates = total;
            this.executions = executions;
            this.runtime = timeTaken;
            this.batches = batches;
            this.failedBatches = failedBatches;
            this.batchErrors = batchErrors;
            this.failedCommits = failedCommits;
            this.commitErrors = commitErrors;
            this.wasTerminated = wasTerminated;
        }
    }

    private class Countdown
    implements Runnable {
        private final String name;
        private final String statement;
        private final long delay;
        private final transient Log log;

        public Countdown(String name, String statement, long delay, Log log) {
            this.name = name;
            this.statement = statement;
            this.delay = delay;
            this.log = log;
        }

        @Override
        public void run() {
            if (Periodic.this.executeNumericResultStatement(this.statement, Collections.emptyMap()) > 0L) {
                Periodic.this.pools.getScheduledExecutorService().schedule(() -> PeriodicUtils.submitJob(this.name, this, this.log, Periodic.this.pools), this.delay, TimeUnit.SECONDS);
            }
        }
    }
}

