/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.server.queryapi;

import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.ConcurrentModificationException;
import java.util.Set;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.core.Response;
import org.neo4j.driver.AuthToken;
import org.neo4j.driver.AuthTokens;
import org.neo4j.driver.Bookmark;
import org.neo4j.driver.Driver;
import org.neo4j.driver.Result;
import org.neo4j.driver.Session;
import org.neo4j.driver.SessionConfig;
import org.neo4j.driver.TransactionConfig;
import org.neo4j.driver.exceptions.Neo4jException;
import org.neo4j.kernel.api.exceptions.Status;
import org.neo4j.logging.InternalLog;
import org.neo4j.logging.InternalLogProvider;
import org.neo4j.server.queryapi.request.AccessMode;
import org.neo4j.server.queryapi.request.AutoCommitResultContainer;
import org.neo4j.server.queryapi.request.QueryRequest;
import org.neo4j.server.queryapi.request.TxManagedResultContainer;
import org.neo4j.server.queryapi.response.QueryResponseBookmarks;
import org.neo4j.server.queryapi.response.QueryResponseTxInfo;
import org.neo4j.server.queryapi.response.error.HttpErrorResponse;
import org.neo4j.server.queryapi.tx.Transaction;
import org.neo4j.server.queryapi.tx.TransactionConcurrentAccessException;
import org.neo4j.server.queryapi.tx.TransactionIdCollisionException;
import org.neo4j.server.queryapi.tx.TransactionManager;
import org.neo4j.server.queryapi.tx.TransactionNotFoundException;
import org.neo4j.server.queryapi.tx.WrongUserException;
import org.neo4j.server.rest.dbms.AuthorizationHeaders;

public class QueryController {
    private final Driver driver;
    private final InternalLog log;
    private final Duration defaultTimeout;
    private final TransactionManager transactionManager;
    private final Integer txIdLength;

    public QueryController(Driver driver, InternalLogProvider logProvider, Duration defaultTimeout, TransactionManager transactionManager, Integer txIdLength) {
        this.transactionManager = transactionManager;
        this.driver = driver;
        this.defaultTimeout = defaultTimeout;
        this.txIdLength = txIdLength;
        this.log = logProvider.getLog(QueryController.class);
    }

    public Response executeQuery(QueryRequest request, HttpServletRequest rawRequest, String databaseName) {
        SessionConfig sessionConfig = this.buildSessionConfig(request, databaseName);
        AuthToken sessionAuthToken = QueryController.extractAuthToken(rawRequest);
        if (sessionAuthToken == null) {
            return Response.status((Response.Status)Response.Status.BAD_REQUEST).build();
        }
        Session session = (Session)this.driver.session(Session.class, sessionConfig, sessionAuthToken);
        try {
            Result result = session.run(request.statement(), request.parameters());
            AutoCommitResultContainer resultContainer = new AutoCommitResultContainer(result, session, request);
            return Response.accepted((Object)resultContainer).build();
        }
        catch (Neo4jException neo4jException) {
            throw neo4jException;
        }
        catch (Exception exception) {
            this.log.error("Local driver failed to execute query", (Throwable)exception);
            throw exception;
        }
    }

    public Response beginTransaction(QueryRequest request, HttpServletRequest rawRequest, String databaseName) {
        SessionConfig sessionConfig = this.buildSessionConfig(request, databaseName);
        String txId = QueryController.randomTxId(this.txIdLength);
        AuthToken sessionAuthToken = QueryController.extractAuthToken(rawRequest);
        if (sessionAuthToken == null) {
            return Response.status((Response.Status)Response.Status.BAD_REQUEST).build();
        }
        Session session = (Session)this.driver.session(Session.class, sessionConfig, sessionAuthToken);
        TxHandling txCleanUpAction = TxHandling.CLOSE;
        try {
            Transaction queryTransaction = this.transactionManager.begin(txId, session, sessionAuthToken, databaseName, this.buildTxConfig(request), request.txType());
            if (request.statement() != null && !request.statement().isEmpty()) {
                queryTransaction.runQuery(request.statement(), request.parameters());
                txCleanUpAction = TxHandling.KEEP_OPEN;
                Response response = QueryController.successWithResultResponse(queryTransaction, request.includeCounters(), false);
                return response;
            }
            txCleanUpAction = TxHandling.RETURN;
            Response response = QueryController.transactionInfoOnlyResponse(queryTransaction);
            return response;
        }
        catch (TransactionIdCollisionException ignored) {
            Response response = QueryController.txCollisionResponse();
            return response;
        }
        catch (Neo4jException neo4jException) {
            throw neo4jException;
        }
        catch (Exception exception) {
            this.log.error("Local driver failed to execute query", (Throwable)exception);
            throw exception;
        }
        finally {
            this.cleanUp(txId, txCleanUpAction);
        }
    }

    public Response continueTransaction(QueryRequest request, HttpServletRequest rawRequest, String databaseName, String txId) {
        return this.executeStatement(request, txId, QueryController.extractAuthToken(rawRequest), databaseName, false);
    }

    public Response commitTransaction(QueryRequest request, HttpServletRequest rawRequest, String databaseName, String txId) {
        return this.executeStatement(request, txId, QueryController.extractAuthToken(rawRequest), databaseName, true);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Response rollbackTransaction(String txId, HttpServletRequest rawRequest, String requestDatabase) {
        Transaction transaction;
        try {
            transaction = this.transactionManager.retrieveTransaction(txId, requestDatabase, QueryController.extractAuthToken(rawRequest));
        }
        catch (TransactionNotFoundException | WrongUserException ex) {
            return QueryController.transactionNotFoundResponse(txId);
        }
        catch (ConcurrentModificationException ex) {
            return QueryController.txConcurrentAccessResponse();
        }
        try {
            transaction.rollback();
        }
        finally {
            this.transactionManager.removeTransaction(txId);
        }
        return Response.ok().build();
    }

    private Response executeStatement(QueryRequest request, String txId, AuthToken requestAuthToken, String requestDatabase, boolean requiresCommit) {
        Transaction queryAPITransaction;
        try {
            queryAPITransaction = this.transactionManager.retrieveTransaction(txId, requestDatabase, requestAuthToken);
        }
        catch (TransactionNotFoundException | WrongUserException ex) {
            return QueryController.transactionNotFoundResponse(txId);
        }
        catch (TransactionConcurrentAccessException ex) {
            return QueryController.txConcurrentAccessResponse();
        }
        TxHandling txCleanUpAction = TxHandling.CLOSE;
        try {
            if (request.statement() != null) {
                queryAPITransaction.runQuery(request.statement(), request.parameters());
                txCleanUpAction = TxHandling.KEEP_OPEN;
                Response response = QueryController.successWithResultResponse(queryAPITransaction, request.includeCounters(), requiresCommit);
                return response;
            }
            if (requiresCommit) {
                Set<Bookmark> bookmarks = queryAPITransaction.commit();
                Response response = QueryController.bookmarksOnlyResponse(bookmarks);
                return response;
            }
            queryAPITransaction.extendTimeout();
            txCleanUpAction = TxHandling.RETURN;
            Response bookmarks = QueryController.transactionInfoOnlyResponse(queryAPITransaction);
            return bookmarks;
        }
        catch (Neo4jException neo4jException) {
            throw neo4jException;
        }
        catch (Exception exception) {
            this.log.error("Local driver failed to execute query", (Throwable)exception);
            throw exception;
        }
        finally {
            this.cleanUp(txId, txCleanUpAction);
        }
    }

    void cleanUp(String txId, TxHandling action) {
        switch (action.ordinal()) {
            case 2: {
                this.transactionManager.removeTransaction(txId);
            }
            case 0: {
                this.transactionManager.releaseTransaction(txId);
            }
        }
    }

    public void closeDriver() {
        this.driver.close();
    }

    private SessionConfig buildSessionConfig(QueryRequest request, String databaseName) {
        SessionConfig.Builder sessionConfigBuilder = SessionConfig.builder().withDatabase(databaseName);
        if (request.bookmarks() != null && !request.bookmarks().isEmpty()) {
            sessionConfigBuilder.withBookmarks((Iterable)request.bookmarks().stream().map(Bookmark::from).collect(Collectors.toList()));
        }
        if (request.impersonatedUser() != null && !request.impersonatedUser().isBlank()) {
            sessionConfigBuilder.withImpersonatedUser(request.impersonatedUser().trim());
        }
        if (request.accessMode() != null) {
            sessionConfigBuilder.withDefaultAccessMode(AccessMode.toDriverAccessMode(request.accessMode()));
        }
        return sessionConfigBuilder.build();
    }

    private TransactionConfig buildTxConfig(QueryRequest request) {
        TransactionConfig.Builder txConfigBuilder = TransactionConfig.builder();
        if (request.maxExecutionTime() > 0) {
            txConfigBuilder.withTimeout(Duration.ofSeconds(request.maxExecutionTime()));
        }
        return txConfigBuilder.build();
    }

    private static AuthToken extractAuthToken(HttpServletRequest request) {
        String authHeader = request.getHeader("Authorization");
        if (authHeader == null) {
            return AuthTokens.none();
        }
        AuthorizationHeaders.ParsedHeader decoded = AuthorizationHeaders.decode(authHeader);
        if (decoded == null) {
            return AuthTokens.none();
        }
        return switch (decoded.scheme()) {
            case AuthorizationHeaders.Scheme.BEARER -> AuthTokens.bearer((String)decoded.values()[0]);
            case AuthorizationHeaders.Scheme.BASIC -> AuthTokens.basic((String)decoded.values()[0], (String)decoded.values()[1]);
            default -> AuthTokens.none();
        };
    }

    private static String randomTxId(Integer length) {
        return Long.toHexString(ThreadLocalRandom.current().nextLong()).substring(0, length);
    }

    private static Response successWithResultResponse(Transaction transaction, boolean requireCounters, boolean requiresCommit) {
        return Response.accepted().entity((Object)new TxManagedResultContainer(transaction, requireCounters, requiresCommit)).build();
    }

    private static Response txConcurrentAccessResponse() {
        return Response.status((Response.Status)Response.Status.BAD_REQUEST).entity((Object)HttpErrorResponse.singleError(Status.Transaction.TransactionAccessedConcurrently.code().serialize(), "Another request is currently accessing this transaction.")).build();
    }

    private static Response bookmarksOnlyResponse(Set<Bookmark> bookmarks) {
        return Response.accepted().entity((Object)QueryResponseBookmarks.fromBookmarks(bookmarks)).build();
    }

    private static Response transactionInfoOnlyResponse(Transaction transaction) {
        return Response.accepted().entity((Object)QueryResponseTxInfo.fromQueryAPITransaction(transaction)).build();
    }

    private static Response transactionNotFoundResponse(String transactionId) {
        return Response.status((int)404).entity((Object)HttpErrorResponse.singleError(Status.Request.Invalid.code().serialize(), String.format("Transaction with Id: \"%s\" was not found. It may have timed out and therefore rolled back or the routing header 'neo4j-cluster-affinity' was not provided.", transactionId))).build();
    }

    private static Response txCollisionResponse() {
        return Response.status((Response.Status)Response.Status.BAD_REQUEST).entity((Object)HttpErrorResponse.singleError(Status.Request.ResourceExhaustion.code().serialize(), "A transaction identifier collision has been detected whilst creating your transaction. Please retry. If this occurs frequently consider increasingthe length of transaction identifier.")).build();
    }

    private Instant generateTimeout() {
        return Instant.now().truncatedTo(ChronoUnit.SECONDS).plus(this.defaultTimeout);
    }

    private static enum TxHandling {
        RETURN,
        KEEP_OPEN,
        CLOSE;

    }
}

