/*
 * Decompiled with CFR 0.152.
 */
package apoc.refactor;

import apoc.Pools;
import apoc.algo.Cover;
import apoc.refactor.NodeRefactorResult;
import apoc.refactor.RefactorRelationshipResult;
import apoc.refactor.UpdatedRelationshipResult;
import apoc.refactor.util.PropertiesManager;
import apoc.refactor.util.RefactorConfig;
import apoc.refactor.util.RefactorUtil;
import apoc.util.Util;
import apoc.util.collection.Iterables;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.function.BiFunction;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;
import org.neo4j.graphdb.Direction;
import org.neo4j.graphdb.Entity;
import org.neo4j.graphdb.GraphDatabaseService;
import org.neo4j.graphdb.Label;
import org.neo4j.graphdb.Node;
import org.neo4j.graphdb.NotFoundException;
import org.neo4j.graphdb.Path;
import org.neo4j.graphdb.Relationship;
import org.neo4j.graphdb.RelationshipType;
import org.neo4j.graphdb.ResourceIterable;
import org.neo4j.graphdb.Result;
import org.neo4j.graphdb.Transaction;
import org.neo4j.graphdb.schema.ConstraintType;
import org.neo4j.internal.kernel.api.procs.ProcedureCallContext;
import org.neo4j.kernel.api.QueryLanguage;
import org.neo4j.kernel.api.procedure.QueryLanguageScope;
import org.neo4j.kernel.impl.coreapi.InternalTransaction;
import org.neo4j.logging.Log;
import org.neo4j.procedure.Context;
import org.neo4j.procedure.Description;
import org.neo4j.procedure.Mode;
import org.neo4j.procedure.Name;
import org.neo4j.procedure.Procedure;

public class GraphRefactoring {
    @Context
    public Transaction tx;
    @Context
    public GraphDatabaseService db;
    @Context
    public Log log;
    @Context
    public Pools pools;
    @Context
    public ProcedureCallContext procedureCallContext;

    @Procedure(name="apoc.refactor.extractNode", mode=Mode.WRITE)
    @Description(value="Expands the given `RELATIONSHIP` VALUES into intermediate `NODE` VALUES.\nThe intermediate `NODE` values are connected by the given `outType` and `inType`.")
    public Stream<NodeRefactorResult> extractNode(@Name(value="rels", description="The relationships to turn into new nodes. Relationships can be of type `STRING` (elementId()), `INTEGER` (id()), `RELATIONSHIP`, or `LIST<STRING | INTEGER | RELATIONSHIP>`.") Object rels, @Name(value="labels", description="The labels to be added to the new nodes.") List<String> labels, @Name(value="outType", description="The type of the outgoing relationship.") String outType, @Name(value="inType", description="The type of the ingoing relationship.") String inType) {
        return Util.relsStream((InternalTransaction)this.tx, rels).map(rel -> {
            NodeRefactorResult result = new NodeRefactorResult(rel.getId());
            try {
                Node copy = Util.withTransactionAndRebind(this.db, this.tx, transaction -> {
                    Node copyNode = RefactorUtil.copyProperties((Entity)rel, transaction.createNode(Util.labels(labels)));
                    copyNode.createRelationshipTo(rel.getEndNode(), RelationshipType.withName((String)outType));
                    return copyNode;
                });
                rel.getStartNode().createRelationshipTo(copy, RelationshipType.withName((String)inType));
                rel.delete();
                return result.withOther(copy);
            }
            catch (Exception e) {
                return result.withError(e);
            }
        });
    }

    @Procedure(name="apoc.refactor.collapseNode", mode=Mode.WRITE)
    @Description(value="Collapses the given `NODE` and replaces it with a `RELATIONSHIP` of the given type.")
    public Stream<UpdatedRelationshipResult> collapseNode(@Name(value="nodes", description="The nodes to collapse. Nodes can be of type `STRING` (elementId()), `INTEGER` (id()), `NODE`, or `LIST<STRING | INTEGER | NODE>`.") Object nodes, @Name(value="relType", description="The name of the resulting relationship type.") String type) {
        return Util.nodeStream((InternalTransaction)this.tx, nodes).map(node -> {
            UpdatedRelationshipResult result = new UpdatedRelationshipResult(node.getId());
            try {
                ResourceIterable outRels = node.getRelationships(Direction.OUTGOING);
                ResourceIterable inRels = node.getRelationships(Direction.INCOMING);
                if (node.getDegree(Direction.OUTGOING) == 1 && node.getDegree(Direction.INCOMING) == 1) {
                    Relationship outRel = (Relationship)outRels.iterator().next();
                    Relationship inRel = (Relationship)inRels.iterator().next();
                    Relationship newRel = inRel.getStartNode().createRelationshipTo(outRel.getEndNode(), RelationshipType.withName((String)type));
                    newRel = RefactorUtil.copyProperties((Entity)node, RefactorUtil.copyProperties((Entity)inRel, RefactorUtil.copyProperties((Entity)outRel, newRel)));
                    for (Relationship r : inRels) {
                        r.delete();
                    }
                    for (Relationship r : outRels) {
                        r.delete();
                    }
                    node.delete();
                    return result.withOther(newRel);
                }
                return result.withError(String.format("Node %d has more that 1 outgoing %d or incoming %d relationships", node.getId(), node.getDegree(Direction.OUTGOING), node.getDegree(Direction.INCOMING)));
            }
            catch (Exception e) {
                return result.withError(e);
            }
        });
    }

    @Procedure(name="apoc.refactor.cloneNodes", mode=Mode.WRITE)
    @Description(value="Clones the given `NODE` values with their labels and properties.\nIt is possible to skip any `NODE` properties using skipProperties (note: this only skips properties on `NODE` values and not their `RELATIONSHIP` values).")
    public Stream<NodeRefactorResult> cloneNodes(@Name(value="nodes", description="The nodes to be cloned.") List<Node> nodes, @Name(value="withRelationships", defaultValue="false", description="Whether or not the connected relationships should also be cloned.") boolean withRelationships, @Name(value="skipProperties", defaultValue="[]", description="Whether or not to skip the node properties when cloning.") List<String> skipProperties) {
        if (nodes == null) {
            return Stream.empty();
        }
        return nodes.stream().map(node -> {
            NodeRefactorResult result = new NodeRefactorResult(node.getId());
            Node newNode = this.tx.createNode(Util.getLabelsArray(node));
            Map properties = node.getAllProperties();
            if (skipProperties != null && !skipProperties.isEmpty()) {
                for (String skip : skipProperties) {
                    properties.remove(skip);
                }
            }
            try {
                RefactorUtil.copyProperties(properties, newNode);
                if (withRelationships) {
                    this.copyRelationships((Node)node, newNode, false, true);
                }
            }
            catch (Exception e) {
                if (withRelationships) {
                    for (Relationship rel : newNode.getRelationships()) {
                        rel.delete();
                    }
                }
                newNode.delete();
                return result.withError(e);
            }
            return result.withOther(newNode);
        });
    }

    @Procedure(name="apoc.refactor.cloneSubgraphFromPaths", mode=Mode.WRITE)
    @Description(value="Clones a sub-graph defined by the given `LIST<PATH>` values.\nIt is possible to skip any `NODE` properties using the `skipProperties` `LIST<STRING>` via the config `MAP`.")
    public Stream<NodeRefactorResult> cloneSubgraphFromPaths(@Name(value="paths", description="The paths to be cloned.") List<Path> paths, @Name(value="config", defaultValue="{}", description="{\n    standinNodes :: LIST<LIST<NODE>>,\n    skipProperties :: LIST<STRING>\n}\n") Map<String, Object> config) {
        if (paths == null || paths.isEmpty()) {
            return Stream.empty();
        }
        HashSet<Node> nodes = new HashSet<Node>();
        HashSet<Relationship> rels = new HashSet<Relationship>();
        for (Path path : paths) {
            for (Relationship rel : path.relationships()) {
                rels.add(rel);
            }
            for (Node node : path.nodes()) {
                nodes.add(node);
            }
        }
        ArrayList<Node> nodesList = new ArrayList<Node>(nodes);
        ArrayList<Relationship> relsList = new ArrayList<Relationship>(rels);
        return this.cloneSubgraph(nodesList, relsList, config);
    }

    @Procedure(name="apoc.refactor.cloneSubgraph", mode=Mode.WRITE)
    @Description(value="Clones the given `NODE` values with their labels and properties (optionally skipping any properties in the `skipProperties` `LIST<STRING>` via the config `MAP`), and clones the given `RELATIONSHIP` values.\nIf no `RELATIONSHIP` values are provided, all existing `RELATIONSHIP` values between the given `NODE` values will be cloned.")
    public Stream<NodeRefactorResult> cloneSubgraph(@Name(value="nodes", description="The nodes to be cloned.") List<Node> nodes, @Name(value="rels", defaultValue="[]", description="The relationships to be cloned. If left empty all relationships between the given nodes will be cloned.") List<Relationship> rels, @Name(value="config", defaultValue="{}", description="{\n    standinNodes :: LIST<LIST<NODE>>,\n    skipProperties :: LIST<STRING>,\n    createNodesInNewTransactions = false :: BOOLEAN\n}\n") Map<String, Object> config) {
        if (nodes == null || nodes.isEmpty()) {
            return Stream.empty();
        }
        HashMap<Node, Node> newNodeByOldNode = new HashMap<Node, Node>(nodes.size());
        ArrayList<NodeRefactorResult> resultStream = new ArrayList<NodeRefactorResult>();
        Map<Node, Node> standinMap = this.asNodePairs(config.get("standinNodes"));
        Set<String> skipProps = GraphRefactoring.asStringSet(config.get("skipProperties"));
        boolean createNodesInInnerTx = Boolean.TRUE.equals(config.getOrDefault("createNodesInNewTransactions", false));
        for (Node oldNode : nodes) {
            if (oldNode == null || standinMap.containsKey(oldNode)) continue;
            NodeRefactorResult result = new NodeRefactorResult(oldNode.getId());
            try {
                Node newNode = !createNodesInInnerTx ? GraphRefactoring.cloneNode(this.tx, oldNode, skipProps) : Util.withTransactionAndRebind(this.db, this.tx, innerTx -> GraphRefactoring.cloneNode(innerTx, oldNode, skipProps));
                resultStream.add(result.withOther(newNode));
                newNodeByOldNode.put(oldNode, newNode);
            }
            catch (Exception e) {
                resultStream.add(result.withError(e));
            }
        }
        Iterator<Object> relsIterator = rels == null || rels.isEmpty() ? Cover.coverNodes(nodes).iterator() : rels.iterator();
        while (relsIterator.hasNext()) {
            Relationship rel = (Relationship)relsIterator.next();
            if (rel == null) continue;
            Node oldStart = rel.getStartNode();
            Node newStart = standinMap.getOrDefault(oldStart, (Node)newNodeByOldNode.get(oldStart));
            Node oldEnd = rel.getEndNode();
            Node newEnd = standinMap.getOrDefault(oldEnd, (Node)newNodeByOldNode.get(oldEnd));
            if (newStart == null || newEnd == null) continue;
            GraphRefactoring.cloneRel(rel, newStart, newEnd, skipProps);
        }
        return resultStream.stream();
    }

    private static Node cloneNode(Transaction tx, Node node, Set<String> skipProps) {
        Node newNode = tx.createNode((Label[])StreamSupport.stream(node.getLabels().spliterator(), false).toArray(Label[]::new));
        try {
            node.getAllProperties().forEach((k, v) -> {
                if (skipProps.isEmpty() || !skipProps.contains(k)) {
                    newNode.setProperty(k, v);
                }
            });
        }
        catch (Exception e) {
            newNode.delete();
            throw e;
        }
        return newNode;
    }

    private static void cloneRel(Relationship base, Node from, Node to, Set<String> skipProps) {
        Relationship rel = from.createRelationshipTo(to, base.getType());
        base.getAllProperties().forEach((k, v) -> {
            if (skipProps.isEmpty() || !skipProps.contains(k)) {
                rel.setProperty(k, v);
            }
        });
    }

    private Map<Node, Node> asNodePairs(Object o) {
        if (o == null) {
            return Collections.emptyMap();
        }
        if (o instanceof List) {
            List list = (List)o;
            return list.stream().filter(Objects::nonNull).map(GraphRefactoring::castNodePair).collect(Collectors.toUnmodifiableMap(l -> (Node)l.get(0), l -> (Node)l.get(1)));
        }
        throw new IllegalArgumentException("Expected a list of node pairs but got " + String.valueOf(o));
    }

    private static Set<String> asStringSet(Object o) {
        Collection c;
        if (o == null) {
            return Collections.emptySet();
        }
        if (o instanceof Collection && (c = (Collection)o).stream().allMatch(i -> i instanceof String)) {
            return c.stream().map(Object::toString).collect(Collectors.toSet());
        }
        throw new IllegalArgumentException("Expected a list of string parameter keys but got " + String.valueOf(o));
    }

    private static List<Node> castNodePair(Object o) {
        List l;
        if (o instanceof List && (l = (List)o).size() == 2 && l.get(0) instanceof Node && l.get(1) instanceof Node) {
            return l;
        }
        throw new IllegalArgumentException("Expected pair of nodes but got " + String.valueOf(o));
    }

    @Procedure(name="apoc.refactor.mergeNodes", mode=Mode.WRITE, eager=true)
    @Description(value="Merges the given `LIST<NODE>` onto the first `NODE` in the `LIST<NODE>`.\nAll `RELATIONSHIP` values are merged onto that `NODE` as well.")
    public Stream<MergedNodeResult> mergeNodes(@Name(value="nodes", description="The nodes to be merged onto the first node.") List<Node> nodes, @Name(value="config", defaultValue="{}", description="{\n    mergeRels :: BOOLEAN,\n    selfRef :: BOOLEAN,\n    produceSelfRef = true :: BOOLEAN,\n    preserveExistingSelfRels = true :: BOOLEAN,\n    countMerge = true :: BOOLEAN,\n    collapsedLabel :: BOOLEAN,\n    singleElementAsArray = false :: BOOLEAN,\n    avoidDuplicates = false :: BOOLEAN,\n    relationshipSelectionStrategy = \"incoming\" :: [\"incoming\", \"outgoing\", \"merge\"]\n    properties :: [\"overwrite\", \"\"discard\", \"combine\"]\n}\n") Map<String, Object> config) {
        if (nodes == null || nodes.isEmpty()) {
            return Stream.empty();
        }
        RefactorConfig conf = new RefactorConfig(config);
        LinkedHashSet<Node> nodesSet = new LinkedHashSet<Node>(nodes);
        nodesSet.stream().sorted(Comparator.comparing(Entity::getElementId)).forEach(arg_0 -> ((Transaction)this.tx).acquireWriteLock(arg_0));
        Node first = nodes.get(0);
        List existingSelfRelIds = conf.isPreservingExistingSelfRels() ? StreamSupport.stream(first.getRelationships().spliterator(), false).filter(Util::isSelfRel).map(Entity::getElementId).collect(Collectors.toList()) : Collections.emptyList();
        nodesSet.stream().skip(1L).forEach(node -> this.mergeNodes((Node)node, first, conf, existingSelfRelIds));
        return Stream.of(new MergedNodeResult(first));
    }

    @Procedure(name="apoc.refactor.mergeRelationships", mode=Mode.WRITE)
    @Description(value="Merges the given `LIST<RELATIONSHIP>` onto the first `RELATIONSHIP` in the `LIST<RELATIONSHIP>`.")
    public Stream<MergedRelationshipResult> mergeRelationships(@Name(value="rels", description="The relationships to be merged onto the first relationship.") List<Relationship> relationships, @Name(value="config", defaultValue="{}", description="{\n    mergeRels :: BOOLEAN,\n    selfRef :: BOOLEAN,\n    produceSelfRef = true :: BOOLEAN,\n    preserveExistingSelfRels = true :: BOOLEAN,\n    countMerge = true :: BOOLEAN,\n    collapsedLabel :: BOOLEAN,\n    singleElementAsArray = false :: BOOLEAN,\n    avoidDuplicates = false :: BOOLEAN,\n    relationshipSelectionStrategy = \"incoming\" :: [\"incoming\", \"outgoing\", \"merge\"]\n    properties :: [\"overwrite\", \"discard\", \"combine\"]\n}\n") Map<String, Object> config) {
        if (relationships == null || relationships.isEmpty()) {
            return Stream.empty();
        }
        LinkedHashSet<Relationship> relationshipsSet = new LinkedHashSet<Relationship>(relationships);
        RefactorConfig conf = new RefactorConfig(config);
        Iterator it = relationshipsSet.iterator();
        Relationship first = (Relationship)it.next();
        while (it.hasNext()) {
            Relationship other = (Relationship)it.next();
            if (first.getStartNode().equals((Object)other.getStartNode()) && first.getEndNode().equals((Object)other.getEndNode())) {
                RefactorUtil.mergeRels(other, first, true, conf);
                continue;
            }
            throw new RuntimeException("All Relationships must have the same start and end nodes.");
        }
        return Stream.of(new MergedRelationshipResult(first));
    }

    @Procedure(name="apoc.refactor.setType", mode=Mode.WRITE)
    @QueryLanguageScope(scope={QueryLanguage.CYPHER_5})
    @Description(value="Changes the type of the given `RELATIONSHIP`.")
    public Stream<UpdatedRelationshipResult> setTypeCypher5(@Name(value="rel", description="The relationship to change the type of.") Relationship rel, @Name(value="newType", description="The new type for the relationship.") String newType) {
        return this.setType(rel, newType);
    }

    @Deprecated
    @Procedure(name="apoc.refactor.setType", mode=Mode.WRITE, deprecatedBy="Cypher's dynamic types: `CREATE (from)-[newRel:$(newType)]->(to) SET newRel = properties(oldRel) DELETE oldRel`.")
    @QueryLanguageScope(scope={QueryLanguage.CYPHER_25})
    @Description(value="Changes the type of the given `RELATIONSHIP`.")
    public Stream<UpdatedRelationshipResult> setType(@Name(value="rel", description="The relationship to change the type of.") Relationship rel, @Name(value="newType", description="The new type for the relationship.") String newType) {
        if (rel == null) {
            return Stream.empty();
        }
        UpdatedRelationshipResult result = new UpdatedRelationshipResult(rel.getId());
        try {
            Relationship newRel = rel.getStartNode().createRelationshipTo(rel.getEndNode(), RelationshipType.withName((String)newType));
            RefactorUtil.copyProperties((Entity)rel, newRel);
            rel.delete();
            return Stream.of(result.withOther(newRel));
        }
        catch (Exception e) {
            return Stream.of(result.withError(e));
        }
    }

    @Procedure(name="apoc.refactor.to", mode=Mode.WRITE, eager=true)
    @Description(value="Redirects the given `RELATIONSHIP` to the given end `NODE`.")
    public Stream<UpdatedRelationshipResult> to(@Name(value="rel", description="The relationship to redirect.") Relationship rel, @Name(value="endNode", description="The new end node the relationship should point to.") Node newNode, @Name(value="config", defaultValue="{}", description="{\n    failOnErrors = false :: BOOLEAN\n}\n") Map<String, Object> config) {
        if (config == null) {
            config = Map.of();
        }
        if (rel == null || newNode == null) {
            return Stream.empty();
        }
        boolean failOnErrors = Boolean.TRUE.equals(config.getOrDefault("failOnErrors", false));
        UpdatedRelationshipResult result = new UpdatedRelationshipResult(rel.getId());
        try {
            RelationshipType type = rel.getType();
            Map properties = rel.getAllProperties();
            Node startNode = rel.getStartNode();
            rel.delete();
            Relationship newRel = startNode.createRelationshipTo(newNode, type);
            properties.forEach((arg_0, arg_1) -> ((Relationship)newRel).setProperty(arg_0, arg_1));
            return Stream.of(result.withOther(newRel));
        }
        catch (Exception e) {
            if (failOnErrors) {
                throw e;
            }
            return Stream.of(result.withError(e));
        }
    }

    @Procedure(name="apoc.refactor.invert", mode=Mode.WRITE, eager=true)
    @Description(value="Inverts the direction of the given `RELATIONSHIP`.")
    public Stream<RefactorRelationshipResult> invert(@Name(value="rel", description="The relationship to reverse.") Relationship rel, @Name(value="config", defaultValue="{}", description="{\n    failOnErrors = false :: BOOLEAN\n}\n") Map<String, Object> config) {
        if (config == null) {
            config = Map.of();
        }
        if (rel == null) {
            return Stream.empty();
        }
        boolean failOnErrors = Boolean.TRUE.equals(config.getOrDefault("failOnErrors", false));
        RefactorRelationshipResult result = new RefactorRelationshipResult(rel.getId());
        try {
            RelationshipType type = rel.getType();
            Map properties = rel.getAllProperties();
            Node startNode = rel.getStartNode();
            Node endNode = rel.getEndNode();
            rel.delete();
            Relationship newRel = endNode.createRelationshipTo(startNode, type);
            properties.forEach((arg_0, arg_1) -> ((Relationship)newRel).setProperty(arg_0, arg_1));
            return Stream.of(result.withOther(newRel));
        }
        catch (Exception e) {
            if (failOnErrors) {
                throw e;
            }
            return Stream.of(result.withError(e));
        }
    }

    @Procedure(name="apoc.refactor.from", mode=Mode.WRITE, eager=true)
    @Description(value="Redirects the given `RELATIONSHIP` to the given start `NODE`.")
    public Stream<RefactorRelationshipResult> from(@Name(value="rel", description="The relationship to redirect.") Relationship rel, @Name(value="newNode", description="The node to redirect the given relationship to.") Node newNode, @Name(value="config", defaultValue="{}", description="{\n    failOnErrors = false :: BOOLEAN\n}\n") Map<String, Object> config) {
        if (config == null) {
            config = Map.of();
        }
        if (rel == null || newNode == null) {
            return Stream.empty();
        }
        RefactorRelationshipResult result = new RefactorRelationshipResult(rel.getId());
        boolean failOnErrors = Boolean.TRUE.equals(config.getOrDefault("failOnErrors", false));
        try {
            RelationshipType type = rel.getType();
            Map properties = rel.getAllProperties();
            Node endNode = rel.getEndNode();
            rel.delete();
            Relationship newRel = newNode.createRelationshipTo(endNode, type);
            properties.forEach((arg_0, arg_1) -> ((Relationship)newRel).setProperty(arg_0, arg_1));
            return Stream.of(result.withOther(newRel));
        }
        catch (Exception e) {
            if (failOnErrors) {
                throw e;
            }
            return Stream.of(result.withError(e));
        }
    }

    @Procedure(name="apoc.refactor.normalizeAsBoolean", mode=Mode.WRITE)
    @Description(value="Refactors the given property to a `BOOLEAN`.")
    public void normalizeAsBoolean(@Name(value="entity", description="The node or relationship whose properties will be normalized to booleans.") Object entity, @Name(value="propertyKey", description="The name of the property key to normalize.") String propertyKey, @Name(value="trueValues", description="The possible representations of true values.") List<Object> trueValues, @Name(value="falseValues", description="The possible representations of false values.") List<Object> falseValues) {
        Entity pc;
        Object value;
        if (entity instanceof Entity && (value = (pc = (Entity)entity).getProperty(propertyKey, null)) != null) {
            boolean isTrue = trueValues.contains(value);
            boolean isFalse = falseValues.contains(value);
            if (isTrue && !isFalse) {
                pc.setProperty(propertyKey, (Object)true);
            }
            if (!isTrue && isFalse) {
                pc.setProperty(propertyKey, (Object)false);
            }
            if (!isTrue && !isFalse) {
                pc.removeProperty(propertyKey);
            }
        }
    }

    @Procedure(name="apoc.refactor.categorize", mode=Mode.WRITE)
    @QueryLanguageScope(scope={QueryLanguage.CYPHER_5})
    @Description(value="Creates new category `NODE` values from `NODE` values in the graph with the specified `sourceKey` as one of its property keys.\nThe new category `NODE` values are then connected to the original `NODE` values with a `RELATIONSHIP` of the given type.")
    public void categorizeCypher5(@Name(value="sourceKey", description="The property key to add to the on the new node.") String sourceKey, @Name(value="type", description="The relationship type to connect to the new node.") String relationshipType, @Name(value="outgoing", description="Whether the relationship should be outgoing or not.") Boolean outgoing, @Name(value="label", description="The label of the new node.") String label, @Name(value="targetKey", description="The name by which the source key value will be referenced on the new node.") String targetKey, @Name(value="copiedKeys", description="A list of additional property keys to be copied to the new node.") List<String> copiedKeys, @Name(value="batchSize", description="The max size of each batch.") long batchSize) throws ExecutionException {
        this.categorize(sourceKey, relationshipType, outgoing, label, targetKey, copiedKeys, batchSize);
    }

    @Deprecated
    @Procedure(name="apoc.refactor.categorize", mode=Mode.WRITE, deprecatedBy="Cypher's CALL {} IN TRANSACTIONS and dynamic labels, see the docs for more information.")
    @QueryLanguageScope(scope={QueryLanguage.CYPHER_25})
    @Description(value="Creates new category `NODE` values from `NODE` values in the graph with the specified `sourceKey` as one of its property keys.\nThe new category `NODE` values are then connected to the original `NODE` values with a `RELATIONSHIP` of the given type.")
    public void categorize(@Name(value="sourceKey", description="The property key to add to the on the new node.") String sourceKey, @Name(value="type", description="The relationship type to connect to the new node.") String relationshipType, @Name(value="outgoing", description="Whether the relationship should be outgoing or not.") Boolean outgoing, @Name(value="label", description="The label of the new node.") String label, @Name(value="targetKey", description="The name by which the source key value will be referenced on the new node.") String targetKey, @Name(value="copiedKeys", description="A list of additional property keys to be copied to the new node.") List<String> copiedKeys, @Name(value="batchSize", description="The max size of each batch.") long batchSize) throws ExecutionException {
        if (sourceKey == null) {
            throw new IllegalArgumentException("Invalid (null) sourceKey");
        }
        if (targetKey == null) {
            throw new IllegalArgumentException("Invalid (null) targetKey");
        }
        copiedKeys.remove(targetKey);
        if (!this.isUniqueConstraintDefinedFor(label, targetKey)) {
            throw new IllegalArgumentException("Before execute this procedure you must define an unique constraint for the label and the targetKey:\n" + String.format("CREATE CONSTRAINT FOR (n:`%s`) REQUIRE n.`%s` IS UNIQUE", label, targetKey));
        }
        ArrayList<Node> batch = null;
        ArrayList<Future<Void>> futures = new ArrayList<Future<Void>>();
        for (Node node : this.tx.getAllNodes()) {
            if (batch == null) {
                batch = new ArrayList<Node>((int)batchSize);
            }
            batch.add(node);
            if ((long)batch.size() != batchSize) continue;
            futures.add(this.categorizeNodes(batch, sourceKey, relationshipType, outgoing, label, targetKey, copiedKeys));
            batch = null;
        }
        if (batch != null) {
            futures.add(this.categorizeNodes(batch, sourceKey, relationshipType, outgoing, label, targetKey, copiedKeys));
        }
        for (Future future : futures) {
            Pools.force(future);
        }
    }

    @Procedure(name="apoc.refactor.deleteAndReconnect", mode=Mode.WRITE, deprecatedBy="Deprecated for removal without a direct replacement, use plain Cypher or create a custom procedure.")
    @Deprecated
    @Description(value="Removes the given `NODE` values from the `PATH` (and graph, including all of its relationships) and reconnects the remaining `NODE` values.\nNote, undefined behaviour for paths that visits the same node multiple times.\nNote, nodes that are not connected in the same direction as the path will not be reconnected, for example `MATCH p=(:A)-->(b:B)<--(:C) CALL apoc.refactor.deleteAndReconnect(p, [b]) ...` will not reconnect the :A and :C nodes.")
    @QueryLanguageScope(scope={QueryLanguage.CYPHER_25})
    public Stream<RefactorGraphResult> deleteAndReconnectCypher25(@Name(value="path", description="The path containing the nodes to delete and the remaining nodes to reconnect.") Path path, @Name(value="nodes", description="The nodes to delete.") List<Node> nodesToRemove, @Name(value="config", defaultValue="{}", description="{\n    relationshipSelectionStrategy = \"incoming\" :: [\"incoming\", \"outgoing\", \"merge\"]\n    properties :: [\"overwrite\", \"discard\", \"combine\"]\n}\n") Map<String, Object> config) {
        return this.deleteAndReconnectCypher5(path, nodesToRemove, config);
    }

    @Procedure(name="apoc.refactor.deleteAndReconnect", mode=Mode.WRITE)
    @Description(value="Removes the given `NODE` values from the `PATH` (and graph, including all of its relationships) and reconnects the remaining `NODE` values.\nNote, undefined behaviour for paths that visits the same node multiple times.\nNote, nodes that are not connected in the same direction as the path will not be reconnected, for example `MATCH p=(:A)-->(b:B)<--(:C) CALL apoc.refactor.deleteAndReconnect(p, [b]) ...` will not reconnect the :A and :C nodes.")
    @QueryLanguageScope(scope={QueryLanguage.CYPHER_5})
    public Stream<RefactorGraphResult> deleteAndReconnectCypher5(@Name(value="path", description="The path containing the nodes to delete and the remaining nodes to reconnect.") Path path, @Name(value="nodes", description="The nodes to delete.") List<Node> nodesToRemove, @Name(value="config", defaultValue="{}", description="{\n    relationshipSelectionStrategy = \"incoming\" :: [\"incoming\", \"outgoing\", \"merge\"]\n    properties :: [\"overwrite\", \"discard\", \"combine\"]\n}\n") Map<String, Object> config) {
        RefactorConfig refactorConfig = new RefactorConfig(config);
        ArrayList<Node> nodes = new ArrayList<Node>();
        path.nodes().forEach(nodes::add);
        Set rels = Iterables.asSet(path.relationships());
        if (!nodes.containsAll(nodesToRemove)) {
            return Stream.empty();
        }
        BiFunction<Node, Direction, Relationship> filterRel = (node, direction) -> StreamSupport.stream(node.getRelationships(direction).spliterator(), false).filter(rels::contains).findFirst().orElse(null);
        nodesToRemove.forEach(node -> {
            Relationship relationshipIn = (Relationship)filterRel.apply((Node)node, Direction.INCOMING);
            Relationship relationshipOut = (Relationship)filterRel.apply((Node)node, Direction.OUTGOING);
            if (relationshipIn == null || relationshipOut == null) {
                rels.remove(relationshipIn == null ? relationshipOut : relationshipIn);
            } else {
                RelationshipType newRelType;
                Node nodeIncoming = relationshipIn.getStartNode();
                Node nodeOutgoing = relationshipOut.getEndNode();
                HashMap<String, Object> newRelProps = new HashMap<String, Object>();
                RefactorConfig.RelationshipSelectionStrategy strategy = refactorConfig.getRelationshipSelectionStrategy();
                switch (strategy) {
                    case INCOMING: {
                        newRelType = relationshipIn.getType();
                        newRelProps.putAll(relationshipIn.getAllProperties());
                        break;
                    }
                    case OUTGOING: {
                        newRelType = relationshipOut.getType();
                        newRelProps.putAll(relationshipOut.getAllProperties());
                        break;
                    }
                    default: {
                        newRelType = RelationshipType.withName((String)(String.valueOf(relationshipIn.getType()) + "_" + String.valueOf(relationshipOut.getType())));
                        newRelProps.putAll(relationshipIn.getAllProperties());
                    }
                }
                Relationship relCreated = nodeIncoming.createRelationshipTo(nodeOutgoing, newRelType);
                newRelProps.forEach((arg_0, arg_1) -> ((Relationship)relCreated).setProperty(arg_0, arg_1));
                if (strategy == RefactorConfig.RelationshipSelectionStrategy.MERGE) {
                    PropertiesManager.mergeProperties(relationshipOut.getAllProperties(), (Entity)relCreated, refactorConfig);
                }
                rels.add(relCreated);
                rels.removeAll(List.of(relationshipIn, relationshipOut));
            }
            this.tx.execute(Util.prefixQuery(this.procedureCallContext, "WITH $node as n DETACH DELETE n"), Map.of("node", node));
            nodes.remove(node);
        });
        return Stream.of(new RefactorGraphResult(nodes, List.copyOf(rels)));
    }

    private boolean isUniqueConstraintDefinedFor(String label, String key) {
        return StreamSupport.stream(this.tx.schema().getConstraints(Label.label((String)label)).spliterator(), false).anyMatch(c -> {
            if (!c.isConstraintType(ConstraintType.UNIQUENESS)) {
                return false;
            }
            return StreamSupport.stream(c.getPropertyKeys().spliterator(), false).allMatch(k -> k.equals(key));
        });
    }

    private Future<Void> categorizeNodes(List<Node> batch, String sourceKey, String relationshipType, Boolean outgoing, String label, String targetKey, List<String> copiedKeys) {
        return this.pools.processBatch(batch, this.db, (innerTx, node) -> {
            Object value = (node = Util.rebind(innerTx, node)).getProperty(sourceKey, null);
            if (value != null) {
                String nodeLabel = Util.sanitize(label);
                String key = Util.sanitize(targetKey);
                String relType = Util.sanitize(relationshipType);
                String q = "WITH $node AS n MERGE (cat:`" + nodeLabel + "` {`" + key + "`: $value}) " + (outgoing != false ? "MERGE (n)-[:`" + relType + "`]->(cat) " : "MERGE (n)<-[:`" + relType + "`]-(cat) ") + "RETURN cat";
                HashMap<String, Object> params = new HashMap<String, Object>(2);
                params.put("node", node);
                params.put("value", value);
                Result result = innerTx.execute(Util.prefixQuery(this.procedureCallContext, q), params);
                if (result.hasNext()) {
                    Node cat = (Node)result.next().get("cat");
                    for (String copiedKey : copiedKeys) {
                        Object copiedValue = node.getProperty(copiedKey, null);
                        if (copiedValue == null) continue;
                        Object catValue = cat.getProperty(copiedKey, null);
                        if (catValue == null) {
                            cat.setProperty(copiedKey, copiedValue);
                            node.removeProperty(copiedKey);
                            continue;
                        }
                        if (!copiedValue.equals(catValue)) continue;
                        node.removeProperty(copiedKey);
                    }
                }
                assert (!result.hasNext());
                result.close();
                node.removeProperty(sourceKey);
            }
        });
    }

    private void mergeNodes(Node source, Node target, RefactorConfig conf, List<String> excludeRelIds) {
        try {
            Map properties = source.getAllProperties();
            Iterable labels = source.getLabels();
            this.copyRelationships(source, target, true, conf.isCreatingNewSelfRel());
            if (conf.getMergeRelsAllowed()) {
                RefactorUtil.mergeRelationshipsWithSameTypeAndDirection(target, conf, Direction.OUTGOING, excludeRelIds);
                RefactorUtil.mergeRelationshipsWithSameTypeAndDirection(target, conf, Direction.INCOMING, excludeRelIds);
            }
            source.delete();
            labels.forEach(arg_0 -> ((Node)target).addLabel(arg_0));
            PropertiesManager.mergeProperties(properties, (Entity)target, conf);
        }
        catch (NotFoundException e) {
            this.log.warn("skipping a node for merging: " + e.getCause().getMessage());
        }
    }

    private void copyRelationships(Node source, Node target, boolean delete, boolean createNewSelfRel) {
        for (Relationship rel : source.getRelationships()) {
            Node startNode = rel.getStartNode();
            Node endNode = rel.getEndNode();
            if (!createNewSelfRel && startNode.getElementId().equals(endNode.getElementId())) {
                if (!delete) continue;
                rel.delete();
                continue;
            }
            if (startNode.getElementId().equals(source.getElementId())) {
                startNode = target;
            }
            if (endNode.getElementId().equals(source.getElementId())) {
                endNode = target;
            }
            RelationshipType type = rel.getType();
            Map properties = rel.getAllProperties();
            if (delete) {
                rel.delete();
            }
            Relationship newRel = startNode.createRelationshipTo(endNode, type);
            properties.forEach((arg_0, arg_1) -> ((Relationship)newRel).setProperty(arg_0, arg_1));
        }
    }

    public record MergedNodeResult(@Description(value="The merged node.") Node node) {
    }

    public record MergedRelationshipResult(@Description(value="The merged relationship.") Relationship rel) {
    }

    public record RefactorGraphResult(@Description(value="The remaining nodes.") List<Node> nodes, @Description(value="The new connecting relationships.") List<Relationship> relationships) {
    }
}

