/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.ssl.config;

import io.netty.handler.ssl.SslProvider;
import io.netty.handler.ssl.util.InsecureTrustManagerFactory;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.DirectoryStream;
import java.nio.file.Path;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.cert.CRLException;
import java.security.cert.CertSelector;
import java.security.cert.CertStore;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateNotYetValidException;
import java.security.cert.CollectionCertStoreParameters;
import java.security.cert.PKIXBuilderParameters;
import java.security.cert.X509CRL;
import java.security.cert.X509CertSelector;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import javax.net.ssl.CertPathTrustManagerParameters;
import javax.net.ssl.TrustManagerFactory;
import org.neo4j.configuration.Config;
import org.neo4j.configuration.SslSystemInternalSettings;
import org.neo4j.configuration.SslSystemSettings;
import org.neo4j.configuration.ssl.ClientAuth;
import org.neo4j.configuration.ssl.SslPolicyConfig;
import org.neo4j.configuration.ssl.SslPolicyScope;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.logging.InternalLog;
import org.neo4j.logging.InternalLogProvider;
import org.neo4j.pki.PkiUtils;
import org.neo4j.ssl.SslPolicy;
import org.neo4j.string.SecureString;

public class SslPolicyLoader {
    private static final DirectoryStream.Filter<Path> ALL_FILES = path -> true;
    private static final DirectoryStream.Filter<Path> SKIP_DOT_FILES = path -> !path.getFileName().toString().startsWith(".");
    private final Map<SslPolicyScope, SslPolicy> policies = new ConcurrentHashMap<SslPolicyScope, SslPolicy>();
    private final Config config;
    private final SslProvider sslProvider;
    private final InternalLogProvider logProvider;
    private final InternalLog log;
    private final boolean skipDotFiles;
    private final FileSystemAbstraction fileSystem;

    private SslPolicyLoader(FileSystemAbstraction fileSystem, Config config, InternalLogProvider logProvider) {
        this.fileSystem = fileSystem;
        this.config = config;
        this.skipDotFiles = (Boolean)config.get(SslSystemInternalSettings.ignore_dotfiles);
        this.sslProvider = (SslProvider)config.get(SslSystemSettings.netty_ssl_provider);
        this.logProvider = logProvider;
        this.log = logProvider.getLog(SslPolicyLoader.class);
    }

    public static SslPolicyLoader create(FileSystemAbstraction fileSystem, Config config, InternalLogProvider logProvider) {
        SslPolicyLoader policyFactory = new SslPolicyLoader(fileSystem, config, logProvider);
        policyFactory.load();
        return policyFactory;
    }

    public SslPolicy getPolicy(SslPolicyScope scope) {
        if (scope == null) {
            return null;
        }
        SslPolicy sslPolicy = this.policies.get(scope);
        if (sslPolicy == null) {
            throw new IllegalArgumentException(String.format("Cannot find enabled SSL policy with name '%s' in the configuration", scope));
        }
        return sslPolicy;
    }

    public boolean hasPolicyForSource(SslPolicyScope scope) {
        return this.policies.containsKey(scope);
    }

    private void load() {
        this.config.getGroups(SslPolicyConfig.class).values().forEach(this::addPolicy);
        this.policies.forEach((scope, sslPolicy) -> this.log.info(String.format("Loaded SSL policy '%s' = %s", scope.name(), sslPolicy)));
    }

    private void addPolicy(SslPolicyConfig policyConfig) {
        SslPolicy policy;
        SslPolicyScope scope;
        if (((Boolean)this.config.get(policyConfig.enabled)).booleanValue() && this.policies.put(scope = policyConfig.getScope(), policy = this.createSslPolicy(policyConfig)) != null) {
            throw new IllegalStateException("Can not have multiple SslPolicies configured for " + scope.name());
        }
    }

    private SslPolicy createSslPolicy(SslPolicyConfig policyConfig) {
        TrustManagerFactory trustManagerFactory;
        Path baseDirectory = (Path)this.config.get(policyConfig.base_directory);
        Path revokedCertificatesDir = (Path)this.config.get(policyConfig.revoked_dir);
        Path privateKeyFile = (Path)this.config.get(policyConfig.private_key);
        Path certificateFile = (Path)this.config.get(policyConfig.public_certificate);
        if (!this.fileSystem.fileExists(baseDirectory)) {
            throw new IllegalArgumentException(String.format("Base directory '%s' for SSL policy with name '%s' does not exist.", baseDirectory, policyConfig.name()));
        }
        KeyAndChain keyAndChain = this.pemKeyAndChain(policyConfig);
        Collection<X509CRL> crls = SslPolicyLoader.getCRLs(this.fileSystem, revokedCertificatesDir, this.certificateFilenameFilter());
        try {
            trustManagerFactory = SslPolicyLoader.createTrustManagerFactory((Boolean)this.config.get(policyConfig.trust_all), crls, keyAndChain.trustStore);
        }
        catch (Exception e) {
            throw new RuntimeException("Failed to create trust manager", e);
        }
        boolean verifyHostname = (Boolean)this.config.get(policyConfig.verify_hostname);
        boolean verifyExpiration = (Boolean)this.config.get(policyConfig.trust_expired) == false;
        ClientAuth clientAuth = (ClientAuth)this.config.get(policyConfig.client_auth);
        List tlsVersions = (List)this.config.get(policyConfig.tls_versions);
        List ciphers = (List)this.config.get(policyConfig.ciphers);
        return new SslPolicy(keyAndChain.privateKey, privateKeyFile, keyAndChain.keyCertChain, certificateFile, tlsVersions, ciphers, clientAuth, trustManagerFactory, this.sslProvider, verifyHostname, verifyExpiration, this.logProvider);
    }

    private KeyAndChain pemKeyAndChain(SslPolicyConfig policyConfig) {
        KeyStore trustStore;
        Path privateKeyFile = (Path)this.config.get(policyConfig.private_key);
        SecureString privateKeyPassword = (SecureString)this.config.get(policyConfig.private_key_password);
        Path trustedCertificatesDir = (Path)this.config.get(policyConfig.trusted_dir);
        ClientAuth clientAuth = (ClientAuth)this.config.get(policyConfig.client_auth);
        try {
            trustStore = this.createTrustStoreFromPem(trustedCertificatesDir, clientAuth);
        }
        catch (Exception e) {
            throw new RuntimeException("Failed to create trust manager based on: " + String.valueOf(trustedCertificatesDir), e);
        }
        if (policyConfig.getScope().isClientOnly() && !this.fileSystem.fileExists(privateKeyFile)) {
            return new KeyAndChain(null, new X509Certificate[0], trustStore);
        }
        Path keyCertChainFile = (Path)this.config.get(policyConfig.public_certificate);
        PrivateKey privateKey = SslPolicyLoader.loadPrivateKey(this.fileSystem, privateKeyFile, privateKeyPassword);
        X509Certificate[] keyCertChain = SslPolicyLoader.loadCertificateChain(this.fileSystem, keyCertChainFile);
        if (!((Boolean)this.config.get(policyConfig.trust_expired)).booleanValue()) {
            for (X509Certificate cert : keyCertChain) {
                if (cert == null) continue;
                try {
                    cert.checkValidity();
                }
                catch (CertificateExpiredException | CertificateNotYetValidException e) {
                    throw new RuntimeException("Expired certificate", e);
                }
            }
        }
        return new KeyAndChain(privateKey, keyCertChain, trustStore);
    }

    private static Collection<X509CRL> getCRLs(FileSystemAbstraction fileSystem, Path revokedCertificatesDir, DirectoryStream.Filter<Path> filter) {
        Path[] revocationFiles;
        if (!fileSystem.fileExists(revokedCertificatesDir)) {
            return new ArrayList<X509CRL>();
        }
        try {
            revocationFiles = fileSystem.listFiles(revokedCertificatesDir, filter);
        }
        catch (IOException e) {
            throw new RuntimeException(String.format("Could not find or list files in revoked directory: %s", revokedCertificatesDir), e);
        }
        ArrayList<X509CRL> crls = new ArrayList<X509CRL>();
        for (Path crl : revocationFiles) {
            try (InputStream input = fileSystem.openAsInputStream(crl);){
                crls.addAll(PkiUtils.CERTIFICATE_FACTORY.generateCRLs(input));
            }
            catch (IOException | CRLException e) {
                throw new RuntimeException(String.format("Could not load CRL: %s", crl), e);
            }
        }
        return crls;
    }

    private static X509Certificate[] loadCertificateChain(FileSystemAbstraction fs, Path keyCertChainFile) {
        try {
            return PkiUtils.loadCertificates(fs, keyCertChainFile);
        }
        catch (Exception e) {
            throw new RuntimeException("Failed to load public certificate chain: " + String.valueOf(keyCertChainFile), e);
        }
    }

    private static PrivateKey loadPrivateKey(FileSystemAbstraction fs, Path privateKeyFile, SecureString privateKeyPassword) {
        String password = privateKeyPassword != null ? privateKeyPassword.getString() : null;
        try {
            return PkiUtils.loadPrivateKey(fs, privateKeyFile, password);
        }
        catch (Exception e) {
            throw new RuntimeException("Failed to load private key: " + String.valueOf(privateKeyFile) + (privateKeyPassword == null ? "" : " (using configured password)"), e);
        }
    }

    private static TrustManagerFactory createTrustManagerFactory(boolean trustAll, Collection<X509CRL> crls, KeyStore trustStore) throws Exception {
        if (trustAll) {
            return InsecureTrustManagerFactory.INSTANCE;
        }
        TrustManagerFactory trustManagerFactory = TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
        if (!crls.isEmpty()) {
            PKIXBuilderParameters pkixParamsBuilder = new PKIXBuilderParameters(trustStore, (CertSelector)new X509CertSelector());
            pkixParamsBuilder.setRevocationEnabled(true);
            pkixParamsBuilder.addCertStore(CertStore.getInstance("Collection", new CollectionCertStoreParameters(crls)));
            trustManagerFactory.init(new CertPathTrustManagerParameters(pkixParamsBuilder));
        } else {
            trustManagerFactory.init(trustStore);
        }
        return trustManagerFactory;
    }

    private KeyStore createTrustStoreFromPem(Path trustedCertificatesDir, ClientAuth clientAuth) throws KeyStoreException, IOException, NoSuchAlgorithmException, CertificateException {
        KeyStore trustStore = KeyStore.getInstance(KeyStore.getDefaultType());
        trustStore.load(null, null);
        if (!this.fileSystem.fileExists(trustedCertificatesDir)) {
            return trustStore;
        }
        Path[] trustedCertFiles = this.fileSystem.listFiles(trustedCertificatesDir, this.certificateFilenameFilter());
        if (clientAuth == ClientAuth.REQUIRE && trustedCertFiles.length == 0) {
            throw new RuntimeException(String.format("Client auth is required but no trust anchors found in: %s", trustedCertificatesDir));
        }
        int i = 0;
        for (Path trustedCertFile : trustedCertFiles) {
            try (InputStream input = this.fileSystem.openAsInputStream(trustedCertFile);){
                try {
                    for (Certificate certificate : PkiUtils.CERTIFICATE_FACTORY.generateCertificates(input)) {
                        X509Certificate cert = (X509Certificate)certificate;
                        trustStore.setCertificateEntry(Integer.toString(i++), cert);
                    }
                }
                catch (Exception e) {
                    throw new CertificateException("Error loading certificate file: " + String.valueOf(trustedCertFile), e);
                }
            }
        }
        return trustStore;
    }

    private DirectoryStream.Filter<Path> certificateFilenameFilter() {
        return this.skipDotFiles ? SKIP_DOT_FILES : ALL_FILES;
    }

    private record KeyAndChain(PrivateKey privateKey, X509Certificate[] keyCertChain, KeyStore trustStore) {
    }
}

