# Copyright (c) 2002-2018 "Neo4j,"
# Neo4j Sweden AB [http://neo4j.com]
#
# This file is part of Neo4j.
#
# Neo4j is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


<#
.SYNOPSIS
Retrieves properties about a Neo4j installation

.DESCRIPTION
Retrieves properties about a Neo4j installation and outputs a Neo4j Server object.

.PARAMETER Neo4jHome
The full path to the Neo4j installation.

.EXAMPLE
Get-Neo4jServer -Neo4jHome 'C:\Neo4j'

Retrieves information about the Neo4j installation at C:\Neo4j

.EXAMPLE
'C:\Neo4j' | Get-Neo4jServer

Retrieves information about the Neo4j installation at C:\Neo4j

.EXAMPLE
Get-Neo4jServer

Retrieves information about the Neo4j installation as determined by Get-Neo4jHome

.OUTPUTS
System.Management.Automation.PSCustomObject
This is a Neo4j Server Object

.LINK
Get-Neo4jHome

.NOTES
This function is private to the powershell module

#>
function Get-Neo4jServer
{
  [CmdletBinding(SupportsShouldProcess = $false,ConfirmImpact = 'Low')]
  param(
    [Parameter(Mandatory = $false,ValueFromPipeline = $true)]
    [Alias('Home')]
    [AllowEmptyString()]
    [string]$Neo4jHome = ''
  )

  begin
  {
  }

  process
  {
    # Get and check the Neo4j Home directory
    if (($Neo4jHome -eq '') -or ($Neo4jHome -eq $null))
    {
      Write-Error "Could not detect the Neo4j Home directory"
      return
    }

    if (-not (Test-Path -Path $Neo4jHome))
    {
      Write-Error "$Neo4jHome does not exist"
      return
    }

    # Convert the path specified into an absolute path
    $Neo4jDir = Get-Item $Neo4jHome
    $Neo4jHome = $Neo4jDir.FullName.TrimEnd('\')

    $ConfDir = Get-Neo4jEnv 'NEO4J_CONF'
    if ($ConfDir -eq $null)
    {
      $ConfDir = (Join-Path -Path $Neo4jHome -ChildPath 'conf')
    }

    # Get the information about the server
    $serverProperties = @{
      'Home' = $Neo4jHome;
      'ConfDir' = $ConfDir;
      'LogDir' = (Join-Path -Path $Neo4jHome -ChildPath 'logs');
      'ServerVersion' = '';
      'ServerType' = 'Community';
      'DatabaseMode' = '';
    }

    # Check if the lib dir exists
    $libPath = (Join-Path -Path $Neo4jHome -ChildPath 'lib')
    if (-not (Test-Path -Path $libPath))
    {
      Write-Error "$Neo4jHome is not a valid Neo4j installation.  Missing $libPath"
      return
    }

    # Scan the lib dir...
    Get-ChildItem (Join-Path -Path $Neo4jHome -ChildPath 'lib') | Where-Object { $_.Name -like 'neo4j-server-*.jar' } | ForEach-Object -Process `
       {
      # if neo4j-server-enterprise-<version>.jar exists then this is the enterprise version
      if ($_.Name -like 'neo4j-server-enterprise-*.jar') { $serverProperties.ServerType = 'Enterprise' }

      # Get the server version from the name of the neo4j-server-<version>.jar file
      if ($matches -ne $null) { $matches.Clear() }
      if ($_.Name -match '^neo4j-server-(\d.+)\.jar$') { $serverProperties.ServerVersion = $matches[1] }
    }
    $serverObject = New-Object -TypeName PSCustomObject -Property $serverProperties

    # Validate the object
    if ([string]$serverObject.ServerVersion -eq '') {
      Write-Error "Unable to determine the version of the installation at $Neo4jHome"
      return
    }

    # Get additional settings...
    $setting = (Get-Neo4jSetting -ConfigurationFile 'neo4j.conf' -Name 'dbms.mode' -Neo4jServer $serverObject)
    if ($setting -ne $null) { $serverObject.DatabaseMode = $setting.value }

    # Set process level environment variables
    #  These should mirror the same paths in neo4j-shared.sh
    $dirSettings = @{ 'NEO4J_DATA' = @{ 'config_var' = 'dbms.directories.data'; 'default' = (Join-Path $Neo4jHome 'data') }
      'NEO4J_LIB' = @{ 'config_var' = 'dbms.directories.lib'; 'default' = (Join-Path $Neo4jHome 'lib') }
      'NEO4J_LOGS' = @{ 'config_var' = 'dbms.directories.logs'; 'default' = (Join-Path $Neo4jHome 'logs') }
      'NEO4J_PLUGINS' = @{ 'config_var' = 'dbms.directories.plugins'; 'default' = (Join-Path $Neo4jHome 'plugins') }
      'NEO4J_RUN' = @{ 'config_var' = 'dbms.directories.run'; 'default' = (Join-Path $Neo4jHome 'run') }
    }
    foreach ($name in $dirSettings.Keys)
    {
      $definition = $dirSettings[$name]
      $configured = (Get-Neo4jSetting -ConfigurationFile 'neo4j.conf' -Name $definition['config_var'] -Neo4jServer $serverObject)
      $value = $definition['default']
      if ($configured -ne $null) { $value = $configured.value }

      if ($value -ne $null) {
        if (-not (Test-Path $value -IsValid)) {
          throw "'$value' is not a valid path entry on this system."
        }

        $absolutePathRegex = '(^\\|^/|^[A-Za-z]:)'
        if (-not ($value -match $absolutePathRegex)) {
          $value = (Join-Path -Path $Neo4jHome -ChildPath $value)
        }
      }
      Set-Neo4jEnv $name $value
    }

    # Set log dir on server object and attempt to create it if it doesn't exist
    $serverObject.LogDir = (Get-Neo4jEnv 'NEO4J_LOGS')
    if ($serverObject.LogDir -ne $null) {
      if (-not (Test-Path -PathType Container -Path $serverObject.LogDir)) {
        New-Item -ItemType Directory -Force -ErrorAction SilentlyContinue -Path $serverObject.LogDir | Out-Null
      }
    }

    #  NEO4J_CONF and NEO4J_HOME are used by the Neo4j Admin Tool
    if ((Get-Neo4jEnv 'NEO4J_CONF') -eq $null) { Set-Neo4jEnv "NEO4J_CONF" $ConfDir }
    if ((Get-Neo4jEnv 'NEO4J_HOME') -eq $null) { Set-Neo4jEnv "NEO4J_HOME" $Neo4jHome }

    # Any deprecation warnings
    $WrapperPath = Join-Path -Path $ConfDir -ChildPath 'neo4j-wrapper.conf'
    if (Test-Path -Path $WrapperPath) { Write-Warning "$WrapperPath is deprecated and support for it will be removed in a future version of Neo4j; please move all your settings to neo4j.conf" }

    Write-Output $serverObject
  }

  end
  {
  }
}

# SIG # Begin signature block
# MIIQ4QYJKoZIhvcNAQcCoIIQ0jCCEM4CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCDY0u3SayGihJOx
# NeuWFCSqYbYxAvIms0ohhqCsyULB96CCDcIwggPFMIICraADAgECAgEAMA0GCSqG
# SIb3DQEBCwUAMIGDMQswCQYDVQQGEwJVUzEQMA4GA1UECBMHQXJpem9uYTETMBEG
# A1UEBxMKU2NvdHRzZGFsZTEaMBgGA1UEChMRR29EYWRkeS5jb20sIEluYy4xMTAv
# BgNVBAMTKEdvIERhZGR5IFJvb3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5IC0gRzIw
# HhcNMDkwOTAxMDAwMDAwWhcNMzcxMjMxMjM1OTU5WjCBgzELMAkGA1UEBhMCVVMx
# EDAOBgNVBAgTB0FyaXpvbmExEzARBgNVBAcTClNjb3R0c2RhbGUxGjAYBgNVBAoT
# EUdvRGFkZHkuY29tLCBJbmMuMTEwLwYDVQQDEyhHbyBEYWRkeSBSb290IENlcnRp
# ZmljYXRlIEF1dGhvcml0eSAtIEcyMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIB
# CgKCAQEAv3FiCPH6WTT3G8kYo/eASVjpIoMTpsUgQwE7hPHmhUmfJ+r2hBtOoLTb
# cJjHMgGxBT4HTu70+k8vWTAi56sZVmvigAf88xZ1gDlRe+X5NbZ0TqmNghPktj+p
# A4P6or6KFWp/3gvDthkUBcrqw6gElDtGfDIN8wBmIsiNaW02jBEYt9OyHGC0OPoC
# jM7T3UYH3go+6118yHz7sCtTpJJiaVElBWEaRIGMLKlDliPfrDqBmg4pxRyp6V0e
# tp6eMAo5zvGIgPtLXcwy7IViQyU0AlYnAZG0O3AqP26x6JyIAX2f1PnbU21gnb8s
# 51iruF9G/M7EGwM8CetJMVxpRrPgRwIDAQABo0IwQDAPBgNVHRMBAf8EBTADAQH/
# MA4GA1UdDwEB/wQEAwIBBjAdBgNVHQ4EFgQUOpqFBxBnKLbv9r0FQW4gwZTaD94w
# DQYJKoZIhvcNAQELBQADggEBAJnbXXnV+ZdZZwNh8X47BjF1LaEgjk9lh7T3ppy8
# 2Okv0Nta7s90jHO0OELaBXv4AnW4/aWx1672194Ty1MQfopG0Zf6ty4rEauQsCeA
# +eifWuk3n6vk32yzhRedPdkkT3mRNdZfBOuAg6uaAi21EPTYkMcEc0DtciWgqZ/s
# nqtoEplXxo8SOgmkvUT9BhU3wZvkMqPtOOjYZPMsfhT8Auqfzf8HaBfbIpA4LXqN
# 0VTxaeNfM8p6PXsK48p/Xznl4nW6xXYYM84s8C9Mrfex585PqMSbSlQGxX991QgP
# 4hz+fhe4rF721BayQwkMTfana7SZhGXKeoji4kS+XPfqHPUwggTQMIIDuKADAgEC
# AgEHMA0GCSqGSIb3DQEBCwUAMIGDMQswCQYDVQQGEwJVUzEQMA4GA1UECBMHQXJp
# em9uYTETMBEGA1UEBxMKU2NvdHRzZGFsZTEaMBgGA1UEChMRR29EYWRkeS5jb20s
# IEluYy4xMTAvBgNVBAMTKEdvIERhZGR5IFJvb3QgQ2VydGlmaWNhdGUgQXV0aG9y
# aXR5IC0gRzIwHhcNMTEwNTAzMDcwMDAwWhcNMzEwNTAzMDcwMDAwWjCBtDELMAkG
# A1UEBhMCVVMxEDAOBgNVBAgTB0FyaXpvbmExEzARBgNVBAcTClNjb3R0c2RhbGUx
# GjAYBgNVBAoTEUdvRGFkZHkuY29tLCBJbmMuMS0wKwYDVQQLEyRodHRwOi8vY2Vy
# dHMuZ29kYWRkeS5jb20vcmVwb3NpdG9yeS8xMzAxBgNVBAMTKkdvIERhZGR5IFNl
# Y3VyZSBDZXJ0aWZpY2F0ZSBBdXRob3JpdHkgLSBHMjCCASIwDQYJKoZIhvcNAQEB
# BQADggEPADCCAQoCggEBALngyxDUr3a91JNi6zBkuIEIbMME2WIXji//PmXPj85i
# 5jxSHNoWRUtVq3hrY4NikM4PaWyZyBoUi0zMRTPqiNyeo68r/oBhnXlXxM8u9D8w
# PF1H/JoWvMM3lkFRjhFLVPgovtCMvvAwOB7zsCb4Zkdjbd5xJkePOEdT0UYdtOPc
# AOpFrL28cdmqbwDb280wOnlPX0xH+B3vW8LEnWA7sbJDkdikM07qs9YnT60liqXG
# 9NXQpq50BWRXiLVEVdQtKjo++Li96TIKApRkxBY6UPFKrud5M68MIAd/6N8EOcJp
# AmxjUvp3wRvIdIfIuZMYUFQ1S2lOvDvTSS4f3MHSUvsCAwEAAaOCARowggEWMA8G
# A1UdEwEB/wQFMAMBAf8wDgYDVR0PAQH/BAQDAgEGMB0GA1UdDgQWBBRAwr0njsw0
# gzCiM9f7bLPwtCyAzjAfBgNVHSMEGDAWgBQ6moUHEGcotu/2vQVBbiDBlNoP3jA0
# BggrBgEFBQcBAQQoMCYwJAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmdvZGFkZHku
# Y29tLzA1BgNVHR8ELjAsMCqgKKAmhiRodHRwOi8vY3JsLmdvZGFkZHkuY29tL2dk
# cm9vdC1nMi5jcmwwRgYDVR0gBD8wPTA7BgRVHSAAMDMwMQYIKwYBBQUHAgEWJWh0
# dHBzOi8vY2VydHMuZ29kYWRkeS5jb20vcmVwb3NpdG9yeS8wDQYJKoZIhvcNAQEL
# BQADggEBAAh+bJMQyDi4lqmQS/+hX08E72w+nIgGyVCPpnP3VzEbvrzkL9v4utNb
# 4LTn5nliDgyi12pjczG19ahIpDsILaJdkNe0fCVPEVYwxLZEnXssneVe5u8MYaq/
# 5Cob7oSeuIN9wUPORKcTcA2RH/TIE62DYNnYcqhzJB61rCIOyheJYlhEG6uJJQEA
# D83EG2LbUbTTD1Eqm/S8c/x2zjakzdnYLOqum/UqspDRTXUYij+KQZAjfVtL/qQD
# WJtGssNgYIP4fVBBzsKhkMO77wIv0hVU7kQV2Qqup4oz7bEtdjYm3ATrn/dhHxXc
# h2/uRpYoraEmfQoJpy4Eo428+LwEMAEwggUhMIIECaADAgECAgkAk4hzBsPFjxsw
# DQYJKoZIhvcNAQELBQAwgbQxCzAJBgNVBAYTAlVTMRAwDgYDVQQIEwdBcml6b25h
# MRMwEQYDVQQHEwpTY290dHNkYWxlMRowGAYDVQQKExFHb0RhZGR5LmNvbSwgSW5j
# LjEtMCsGA1UECxMkaHR0cDovL2NlcnRzLmdvZGFkZHkuY29tL3JlcG9zaXRvcnkv
# MTMwMQYDVQQDEypHbyBEYWRkeSBTZWN1cmUgQ2VydGlmaWNhdGUgQXV0aG9yaXR5
# IC0gRzIwHhcNMjAxMTEzMTk0MTM1WhcNMjMxMTEzMTkzNzAzWjBiMQswCQYDVQQG
# EwJVUzETMBEGA1UECBMKQ2FsaWZvcm5pYTESMBAGA1UEBxMJU2FuIE1hdGVvMRQw
# EgYDVQQKEwtOZW80aiwgSW5jLjEUMBIGA1UEAxMLTmVvNGosIEluYy4wggEiMA0G
# CSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCupcKgi9Kd2HEJkPaVeXxXknUE0gA7
# KBMJaFl5jYiL1VzUQvOgy01SAXhYMataV1mDyEOnG/kF5O4HEtw1Rzx8UEb4rMHZ
# COBtpcYN7FHB5X33ujKK2RbfZPdvM60z+enmQjW8vW5USSEjX0yk4xaLeopx41TU
# WwdnnABHEqlizFUrqqBJru5VxzlAEatnGEnjpWAAIsNAbxSq7uHoDEaJimetOKTr
# UgofmP6dfQ74+ggIBVNmJ1Ansx+O4zNFHjLeV+3/sfpwzbXiVi76ZChe95gOeEhW
# cwY9GJD8GQ5E3S+GoCsUwwO9lQL26W6HhaNtjsQ6VYkHPoEv/M+liwazAgMBAAGj
# ggGFMIIBgTAMBgNVHRMBAf8EAjAAMBMGA1UdJQQMMAoGCCsGAQUFBwMDMA4GA1Ud
# DwEB/wQEAwIHgDA1BgNVHR8ELjAsMCqgKKAmhiRodHRwOi8vY3JsLmdvZGFkZHku
# Y29tL2dkaWcyczUtNi5jcmwwXQYDVR0gBFYwVDBIBgtghkgBhv1tAQcXAjA5MDcG
# CCsGAQUFBwIBFitodHRwOi8vY2VydGlmaWNhdGVzLmdvZGFkZHkuY29tL3JlcG9z
# aXRvcnkvMAgGBmeBDAEEATB2BggrBgEFBQcBAQRqMGgwJAYIKwYBBQUHMAGGGGh0
# dHA6Ly9vY3NwLmdvZGFkZHkuY29tLzBABggrBgEFBQcwAoY0aHR0cDovL2NlcnRp
# ZmljYXRlcy5nb2RhZGR5LmNvbS9yZXBvc2l0b3J5L2dkaWcyLmNydDAfBgNVHSME
# GDAWgBRAwr0njsw0gzCiM9f7bLPwtCyAzjAdBgNVHQ4EFgQUviptvnVEufNrrVtz
# Lo74S/A2WqswDQYJKoZIhvcNAQELBQADggEBAF8VvAd5y44ZxAegu+vIPyJpJndK
# n79J1ruEggtr7aWoxLmqx870o8QladSi/cWKw0IAaZN8sJVxR2S7UT/vWLFbM2qe
# haLfK/RixAI5Rd0Lsxiy3m/ugA0g79eQaeYg71jBva/gzE2uEOMPa8YK4oEYmXxz
# LLdZebFqOnn5+QiKLWStEO2FFHU61CE0flokicMUL2V6KHCBpO5MMlYlc1Vmcmvi
# L5xgIIzfmsCDRVV46tKpFZGQjRG77zKwDcJNbi3WRdC3ydJ4vG3D12mIIeNck4K9
# UC5r2psl4vXbz7RdRHfcZqfm0r3AI9SlCVzj9rn6fzRYcn3RT16kQqeiMOkxggJ1
# MIICcQIBATCBwjCBtDELMAkGA1UEBhMCVVMxEDAOBgNVBAgTB0FyaXpvbmExEzAR
# BgNVBAcTClNjb3R0c2RhbGUxGjAYBgNVBAoTEUdvRGFkZHkuY29tLCBJbmMuMS0w
# KwYDVQQLEyRodHRwOi8vY2VydHMuZ29kYWRkeS5jb20vcmVwb3NpdG9yeS8xMzAx
# BgNVBAMTKkdvIERhZGR5IFNlY3VyZSBDZXJ0aWZpY2F0ZSBBdXRob3JpdHkgLSBH
# MgIJAJOIcwbDxY8bMA0GCWCGSAFlAwQCAQUAoIGEMBgGCisGAQQBgjcCAQwxCjAI
# oAKAAKECgAAwGQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGCNwIB
# CzEOMAwGCisGAQQBgjcCARUwLwYJKoZIhvcNAQkEMSIEIFv4ULGewNN9xFrnsoGw
# q7MlEf5Y1tHrQEu1/BxmbeQAMA0GCSqGSIb3DQEBAQUABIIBAG8Bn0CylTjGYiqv
# XklNGq6+OuSdfmU6sJRu2ddpI90Uv2TBvQ+1lDr6xzpUO4s3jvnC9YUF7k0jpr0b
# SzeFmb3vIAA0qOfeu+Vio6ElYlI0jOGMGLiTkMuh9+zeGmKYfSnJCscgcPpYUs4A
# gOzdKLOOq+AZ1buGMbXQIFPEDl0/Fm4c+CuaE9f+0L8sbkK0NFkQNXCD/3VC3GBn
# Y2mvQ07vVY8h9Ncle7rfLWgCJIsOpn5sVIqLvQ4WzK5vWtaI7z7vbma6YZK+KQNi
# kzqYLqLE3JfL/lvBjiyCTtzRWhI9HVOaXWfF89uRI3pTW5jZ40iF8hkNGYH55Rzc
# WnxFmYY=
# SIG # End signature block
