/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.export;

import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.file.Path;
import java.util.zip.CRC32;
import java.util.zip.GZIPInputStream;
import java.util.zip.ZipException;
import org.apache.commons.compress.archivers.tar.TarArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream;
import org.apache.commons.lang3.StringUtils;
import org.neo4j.cli.AbstractAdminCommand;
import org.neo4j.cli.CommandFailedException;
import org.neo4j.cli.Converters;
import org.neo4j.cli.ExecutionContext;
import org.neo4j.dbms.archive.Loader;
import org.neo4j.export.PushToCloudCLI;
import org.neo4j.export.UploadURLFactory;
import org.neo4j.export.aura.AuraClient;
import org.neo4j.export.aura.AuraConsole;
import org.neo4j.export.aura.AuraJsonMapper;
import org.neo4j.export.aura.AuraURLFactory;
import org.neo4j.export.providers.SignedUpload;
import org.neo4j.export.providers.SignedUploadURLFactory;
import org.neo4j.export.util.IOCommon;
import org.neo4j.io.ByteUnit;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.io.fs.StoreChannel;
import org.neo4j.io.memory.NativeScopedBuffer;
import org.neo4j.kernel.database.NormalizedDatabaseName;
import org.neo4j.memory.EmptyMemoryTracker;
import org.neo4j.memory.MemoryTracker;
import picocli.CommandLine;

@CommandLine.Command(name="upload", description={"Push a local database to a Neo4j Aura instance. The target location is a Neo4j Aura Bolt URI. If Neo4j Cloud username and password are not provided either as a command option or as an environment variable, they will be requested interactively "})
public class UploadCommand
extends AbstractAdminCommand {
    private static final long CRC32_BUFFER_SIZE = ByteUnit.mebiBytes((long)4L);
    private static final String DEV_MODE_VAR_NAME = "P2C_DEV_MODE";
    private static final String ENV_NEO4J_USERNAME = "NEO4J_USERNAME";
    private static final String ENV_NEO4J_PASSWORD = "NEO4J_PASSWORD";
    private static final String TO_PASSWORD = "--to-password";
    private final PushToCloudCLI pushToCloudCLI;
    private final AuraClient.AuraClientBuilder clientBuilder;
    private final AuraURLFactory auraURLFactory;
    private final UploadURLFactory uploadURLFactory;
    @CommandLine.Parameters(paramLabel="<database>", description={"Name of the database that should be uploaded. The name is used to select a dump file which is expected to be named <database>.dump."}, converter={Converters.DatabaseNameConverter.class})
    private NormalizedDatabaseName database;
    @CommandLine.Option(names={"--from-path"}, paramLabel="<path>", description={"'/path/to/directory-containing-dump' Path to a directory containing a database dump to upload."}, required=true)
    private Path dumpDirectory;
    @CommandLine.Option(names={"--to-uri"}, paramLabel="<uri>", arity="1", required=true, description={"'neo4j://mydatabaseid.databases.neo4j.io' Bolt URI of the target database."})
    private String boltURI;
    @CommandLine.Option(names={"--to-user"}, defaultValue="${NEO4J_USERNAME}", description={"Username of the target database to push this database to. Prompt will ask for a username if not provided. %nDefault:  The value of the NEO4J_USERNAME environment variable."})
    private String username;
    @CommandLine.Option(names={"--to-password"}, defaultValue="${NEO4J_PASSWORD}", description={"Password of the target database to push this database to. Prompt will ask for a password if not provided. %nDefault:  The value of the NEO4J_PASSWORD environment variable."})
    private String password;
    @CommandLine.Option(names={"--overwrite-destination"}, arity="0..1", paramLabel="true|false", fallbackValue="true", showDefaultValue=CommandLine.Help.Visibility.ALWAYS, description={"Overwrite the data in the target database."})
    private boolean overwrite;
    @CommandLine.Option(names={"--to"}, paramLabel="<destination>", description={"The destination for the upload."}, defaultValue="aura", showDefaultValue=CommandLine.Help.Visibility.ALWAYS)
    private String to;

    public UploadCommand(ExecutionContext ctx, AuraClient.AuraClientBuilder clientBuilder, AuraURLFactory auraURLFactory, UploadURLFactory uploadURLFactory, PushToCloudCLI pushToCloudCLI) {
        super(ctx);
        this.clientBuilder = clientBuilder;
        this.pushToCloudCLI = pushToCloudCLI;
        this.auraURLFactory = auraURLFactory;
        this.uploadURLFactory = uploadURLFactory;
    }

    public static long readSizeFromDumpMetaData(ExecutionContext ctx, Path dump) {
        Loader.DumpMetaData metaData;
        try {
            FileSystemAbstraction fileSystem = ctx.fs();
            metaData = new Loader(fileSystem, System.out).getMetaData(() -> fileSystem.openAsInputStream(dump));
        }
        catch (IOException e) {
            throw new CommandFailedException("Unable to check size of database dump.", (Throwable)e);
        }
        return Long.parseLong(metaData.byteCount());
    }

    public static String sizeText(long size) {
        return String.format("%.1f GB", UploadCommand.bytesToGibibytes(size));
    }

    public static double bytesToGibibytes(long sizeInBytes) {
        return (double)sizeInBytes / 1.073741824E9;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public long readSizeFromTarMetaData(ExecutionContext ctx, Path tar, String dbName) {
        FileSystemAbstraction fileSystem = ctx.fs();
        try (TarArchiveInputStream tais = new TarArchiveInputStream(this.maybeGzipped(tar, fileSystem));){
            TarArchiveEntry entry;
            while ((entry = tais.getNextTarEntry()) != null) {
                if (!entry.getName().endsWith(dbName + ".dump")) continue;
                Loader.DumpMetaData metaData = new Loader(fileSystem, System.out).getMetaData(() -> tais);
                long l = Long.parseLong(metaData.byteCount());
                return l;
            }
            throw new CommandFailedException(String.format("TAR file %s does not contain dump for  database %s", tar, dbName));
        }
        catch (IOException e) {
            throw new CommandFailedException("Unable to check size of tar dump database.", (Throwable)e);
        }
    }

    private InputStream maybeGzipped(Path tar, FileSystemAbstraction fileSystem) throws IOException {
        try {
            return new GZIPInputStream(fileSystem.openAsInputStream(tar));
        }
        catch (ZipException e) {
            return fileSystem.openAsInputStream(tar);
        }
    }

    public void execute() {
        try {
            char[] pass;
            if (!"aura".equals(this.to)) {
                throw new CommandFailedException(String.format("'%s' is not a supported destination. Supported destinations are: 'aura'", this.to));
            }
            if (StringUtils.isBlank((CharSequence)this.username) && StringUtils.isBlank((CharSequence)(this.username = this.pushToCloudCLI.readLine("%s", "Neo4j aura username (default: neo4j):")))) {
                this.username = "neo4j";
            }
            if (StringUtils.isBlank((CharSequence)this.password)) {
                pass = this.pushToCloudCLI.readPassword("Neo4j aura password for %s:", this.username);
                if (pass.length == 0) {
                    throw new CommandFailedException(String.format("Please supply a password, either by '%s' parameter, '%s' environment variable, or prompt", TO_PASSWORD, ENV_NEO4J_PASSWORD));
                }
            } else {
                pass = this.password.toCharArray();
            }
            boolean devMode = this.pushToCloudCLI.readDevMode(DEV_MODE_VAR_NAME);
            AuraConsole auraConsole = this.auraURLFactory.buildConsoleURI(this.boltURI, devMode);
            AuraClient auraClient = this.clientBuilder.withAuraConsole(auraConsole).withUserName(this.username).withPassword(pass).withConsent(this.overwrite).withBoltURI(this.boltURI).withDefaults().build();
            DumpUploader uploader = this.makeDumpUploader(this.dumpDirectory, this.database.name());
            ((Uploader)uploader).process(auraClient);
        }
        catch (Exception e) {
            throw new CommandFailedException(e.getMessage(), (Throwable)e);
        }
    }

    private void verbose(String format, Object ... args) {
        if (this.verbose) {
            this.ctx.out().printf(format, args);
        }
    }

    public DumpUploader makeDumpUploader(Path dump, String database) {
        if (!this.ctx.fs().isDirectory(dump)) {
            throw new CommandFailedException(String.format("The provided source directory '%s' doesn't exist", dump));
        }
        Path dumpFile = dump.resolve(database + ".dump");
        if (!this.ctx.fs().fileExists(dumpFile)) {
            Path tarFile = dump.resolve(database + ".tar");
            if (!this.ctx.fs().fileExists(tarFile)) {
                throw new CommandFailedException(String.format("Dump files '%s' or '%s' do not exist", dumpFile.toAbsolutePath(), tarFile.toAbsolutePath()));
            }
            dumpFile = tarFile;
        }
        return new DumpUploader(new Source(this.ctx.fs(), dumpFile, this.dumpSize(dumpFile, database)));
    }

    private long dumpSize(Path dump, String database) {
        long sizeInBytes = dump.getFileName().toString().endsWith(".dump") ? UploadCommand.readSizeFromDumpMetaData(this.ctx, dump) : this.readSizeFromTarMetaData(this.ctx, dump, database);
        this.verbose("Determined DumpSize=%d bytes from dump at %s\n", sizeInBytes, dump);
        return sizeInBytes;
    }

    class DumpUploader
    extends Uploader {
        DumpUploader(Source source) {
            super(source);
        }

        @Override
        void process(AuraClient auraClient) {
            long crc32Sum;
            String consoleURL = auraClient.getAuraConsole().baseURL();
            UploadCommand.this.verbose("Checking database size %s fits at %s\n", UploadCommand.sizeText(this.size()), consoleURL);
            String bearerToken = auraClient.authenticate(UploadCommand.this.verbose);
            auraClient.checkSize(UploadCommand.this.verbose, this.size(), bearerToken);
            UploadCommand.this.verbose("Uploading data of %s to %s\n", UploadCommand.sizeText(this.size()), consoleURL);
            try {
                crc32Sum = this.source.crc32Sum();
            }
            catch (IOException e) {
                throw new CommandFailedException("Failed to process dump file", (Throwable)e);
            }
            long dumpSize = IOCommon.getFileSize(this.source, UploadCommand.this.ctx);
            AuraJsonMapper.SignedURIBodyResponse signedURIBodyResponse = auraClient.initatePresignedUpload(crc32Sum, dumpSize, this.size(), bearerToken);
            SignedUpload signedUpload = UploadCommand.this.uploadURLFactory.fromAuraResponse(signedURIBodyResponse, UploadCommand.this.ctx, UploadCommand.this.boltURI);
            signedUpload.copy(UploadCommand.this.verbose, this.source);
            try {
                this.triggerImportForDB(auraClient, bearerToken, crc32Sum, signedURIBodyResponse);
                UploadCommand.this.verbose("Polling status\n", new Object[0]);
                auraClient.doStatusPolling(UploadCommand.this.verbose, bearerToken, this.source.size());
            }
            catch (IOException e) {
                throw new CommandFailedException("Failed to trigger import, please contact Aura support", (Throwable)e);
            }
            catch (InterruptedException e) {
                throw new CommandFailedException("Command interrupted", (Throwable)e);
            }
            UploadCommand.this.ctx.out().println("Dump successfully uploaded to Aura");
            UploadCommand.this.ctx.out().println(String.format("Your dump at %s can now be deleted.", this.source.path()));
        }

        private void triggerImportForDB(AuraClient auraClient, String bearerToken, long crc32Sum, AuraJsonMapper.SignedURIBodyResponse signedURIBodyResponse) throws IOException {
            if (signedURIBodyResponse.Provider.equalsIgnoreCase(String.valueOf((Object)SignedUploadURLFactory.Provider.AWS))) {
                AuraJsonMapper.UploadStatusResponse uploadStatusResponse = auraClient.uploadStatus(UploadCommand.this.verbose, crc32Sum, signedURIBodyResponse.UploadID, bearerToken);
                auraClient.triggerAWSImportProtocol(UploadCommand.this.verbose, this.source.path(), crc32Sum, bearerToken, uploadStatusResponse);
            } else {
                auraClient.triggerGCPImportProtocol(UploadCommand.this.verbose, this.source.path(), crc32Sum, bearerToken);
            }
        }
    }

    static abstract class Uploader {
        protected final Source source;

        Uploader(Source source) {
            this.source = source;
        }

        long size() {
            return this.source.size();
        }

        abstract void process(AuraClient var1);
    }

    public record Source(FileSystemAbstraction fs, Path path, long size) {
        long crc32Sum() throws IOException {
            CRC32 crc = new CRC32();
            try (StoreChannel channel = this.fs.read(this.path);
                 NativeScopedBuffer buffer = new NativeScopedBuffer(CRC32_BUFFER_SIZE, ByteOrder.LITTLE_ENDIAN, (MemoryTracker)EmptyMemoryTracker.INSTANCE);){
                ByteBuffer byteBuffer = buffer.getBuffer();
                while (channel.read(byteBuffer) != -1) {
                    byteBuffer.flip();
                    crc.update(byteBuffer);
                    byteBuffer.clear();
                }
            }
            return crc.getValue();
        }
    }
}

